// Dafny program PS-Completeness.dfy compiled into C#
// To recompile, use 'csc' with: /r:System.Numerics.dll
// and choosing /target:exe or /target:library
// You might also want to include compiler switches like:
//     /debug /nowarn:0164 /nowarn:0219 /nowarn:1717

using System;
using System.Numerics;
[assembly: DafnyAssembly.DafnySourceAttribute(@"
// Dafny 2.1.1.10209
// Command Line Options: 
// PS-Completeness.dfy


module PS_Completeness {

  import opened Utils = Utils

  import opened QCSP = QCSP_Instance`{Lemmas_for_PS_Completeness}

  import opened PS = Proof_System`{Lemma_and_Defs}

  export function_methods_for_Implementation
    reveals projection, canonical_judgement, join, dualProjection
    provides Utils, QCSP, PS

  function method projection<T>(j: Judgement<T>, U: set<Name>, phi: Formula, B: Structure<T>): Judgement<T>
    requires wfJudgement(j, phi, B)
    requires U <= j.V
    ensures wfJudgement(projection(j, U, phi, B), phi, B)
    ensures is_projection(projection(j, U, phi, B), j, phi, B)
    decreases j, U, phi, B
  {
    J(j.i, U, set f: map<Name, T> {:trigger projectVal(f, U)} {:trigger f in j.F} | f in j.F :: projectVal(f, U))
  }

  function join<T>(j1: Judgement<T>, j2: Judgement<T>, phi: Formula, B: Structure<T>): Judgement<T>
    requires wfJudgement(j1, phi, B) && wfJudgement(j2, phi, B)
    requires j1.i == j2.i
    ensures wfJudgement(join(j1, j2, phi, B), phi, B)
    ensures is_join(join(j1, j2, phi, B), j1, j2, phi, B)
    decreases j1, j2, phi, B
  {
    J(j1.i, j1.V + j2.V, set f: Valuation<T> {:trigger projectVal(f, j2.V)} {:trigger projectVal(f, j1.V)} | f in allMaps(j1.V + j2.V, B.Dom) && projectVal(f, j1.V) in j1.F && projectVal(f, j2.V) in j2.F)
  }

  function dualProjection<T>(v: Name, j: Judgement<T>, phi: Formula, B: Structure<T>): Judgement<T>
    requires wfJudgement(j, phi, B)
    requires |j.i| > 0 && j.i[..|j.i| - 1] + [0] == j.i
    requires j.i[..|j.i| - 1] in setOfIndex(phi)
    requires FoI(j.i[..|j.i| - 1], phi, B.Sig).Forall? && FoI(j.i[..|j.i| - 1], phi, B.Sig).x == v
    requires v in j.V
    ensures wfJudgement(dualProjection(v, j, phi, B), phi, B)
    ensures is_dualProjection(dualProjection(v, j, phi, B), v, j, phi, B)
    decreases v, j, phi, B
  {
    indexSubformula_Lemma(j.i[..|j.i| - 1], phi, B.Sig);
    J(j.i[..|j.i| - 1], j.V - {v}, set f: Valuation<T> | f in allMaps(j.V - {v}, B.Dom) && forall b: T :: b in B.Dom ==> f[v := b] in j.F)
  }

  function canonical_judgement<T>(i: seq<int>, phi: Formula, B: Structure<T>): Judgement<T>
    requires wfQCSP_Instance(phi, B)
    requires i in setOfIndex(phi)
    ensures canonical_judgement(i, phi, B).i == i
    ensures canonical_judgement(i, phi, B).V == freeVar(FoI(i, phi, B.Sig))
    ensures wfJudgement(canonical_judgement(i, phi, B), phi, B)
    decreases FoI(i, phi, B.Sig)
  {
    var phii: Formula := FoI(i, phi, B.Sig);
    match phii
    case Atom(R, par) =>
      var F := set f: Valuation<T> | f in allMaps(setOf(par), B.Dom) && HOmap(f, par) in B.I[R];
      J(i, setOf(par), F)
    case And(phi0, phi1) =>
      indexSubformula_Lemma(i, phi, B.Sig); var j0' := canonical_judgement(i + [0], phi, B); var j1' := canonical_judgement(i + [1], phi, B); var j0 := J(i, j0'.V, j0'.F); var j1 := J(i, j1'.V, j1'.F); join(j0, j1, phi, B)
    case Forall(x, phik) =>
      indexSubformula_Lemma(i, phi, B.Sig); var j0 := canonical_judgement(i + [0], phi, B); if x in j0.V then dualProjection(x, j0, phi, B) else J(i, j0.V, j0.F)
    case Exists(x, phik) =>
      indexSubformula_Lemma(i, phi, B.Sig);
      var j0 := canonical_judgement(i + [0], phi, B);
      var jp := projection(j0, j0.V - {x}, phi, B);
      if x in j0.V then
        J(i, jp.V, jp.F)
      else
        J(i, j0.V, j0.F)
  }

  function setOfValmodels<T>(phi: Formula, B: Structure<T>): set<Valuation<T>>
    requires wfStructure(B) && wfFormula(B.Sig, phi)
    decreases phi, B
  {
    set f: Valuation<T> {:trigger models(B, f, phi)} {:trigger f in allMaps(freeVar(phi), B.Dom)} | f in allMaps(freeVar(phi), B.Dom) && models(B, f, phi)
  }

  lemma /*{:_induction phi0, phi1}*/ setofValmodelsAnd_Lemma<T>(phi0: Formula, phi1: Formula, B: Structure<T>)
    requires wfStructure(B) && wfFormula(B.Sig, phi0) && wfFormula(B.Sig, phi1)
    ensures setOfValmodels(And(phi0, phi1), B) == set f: Valuation<T> {:trigger projectVal(f, freeVar(phi1))} {:trigger projectVal(f, freeVar(phi0))} | f in allMaps(freeVar(phi0) + freeVar(phi1), B.Dom) && projectVal(f, freeVar(phi0)) in setOfValmodels(phi0, B) && projectVal(f, freeVar(phi1)) in setOfValmodels(phi1, B)
    decreases phi0, phi1, B
  {
    ghost var F := set f: Valuation<T> {:trigger projectVal(f, freeVar(phi1))} {:trigger projectVal(f, freeVar(phi0))} | f in allMaps(freeVar(phi0) + freeVar(phi1), B.Dom) && projectVal(f, freeVar(phi0)) in setOfValmodels(phi0, B) && projectVal(f, freeVar(phi1)) in setOfValmodels(phi1, B);
    forall f: Valuation<T> | f in F
      ensures f in setOfValmodels(And(phi0, phi1), B)
    {
      assert projectVal(f, freeVar(phi0)) in allMaps(freeVar(phi0), B.Dom) && projectVal(f, freeVar(phi0)).Values <= B.Dom && models(B, projectVal(f, freeVar(phi0)), phi0) && projectVal(f, freeVar(phi1)) in allMaps(freeVar(phi1), B.Dom) && projectVal(f, freeVar(phi1)).Values <= B.Dom && models(B, projectVal(f, freeVar(phi1)), phi1);
      modelsAnd_Lemma(B, f, And(phi0, phi1));
      assert f in allMaps(freeVar(And(phi0, phi1)), B.Dom) && f.Values <= B.Dom && models(B, f, And(phi0, phi1));
      assert f in setOfValmodels(And(phi0, phi1), B);
    }
    forall f: Valuation<T> | f in setOfValmodels(And(phi0, phi1), B)
      ensures f in F
    {
      modelsAnd_Lemma(B, f, And(phi0, phi1));
      assert projectVal(f, freeVar(phi0)).Values <= B.Dom && models(B, projectVal(f, freeVar(phi0)), phi0) && projectVal(f, freeVar(phi1)).Values <= B.Dom && models(B, projectVal(f, freeVar(phi1)), phi1);
      allMaps_Correct_Lemma(projectVal(f, freeVar(phi0)), B.Dom);
      assert projectVal(f, freeVar(phi0)) in allMaps(freeVar(phi0), B.Dom);
      assert projectVal(f, freeVar(phi0)) in setOfValmodels(phi0, B);
      allMaps_Correct_Lemma(projectVal(f, freeVar(phi1)), B.Dom);
      assert projectVal(f, freeVar(phi1)) in allMaps(freeVar(phi1), B.Dom);
      assert projectVal(f, freeVar(phi1)) in setOfValmodels(phi1, B);
      assert f in F;
    }
  }

  lemma /*{:_induction beta}*/ setofValmodelsForall_Lemma<T>(x: Name, beta: Formula, B: Structure<T>)
    requires wfStructure(B) && wfFormula(B.Sig, Forall(x, beta))
    requires x in freeVar(beta)
    ensures setOfValmodels(Forall(x, beta), B) == set f: Valuation<T> | f in allMaps(freeVar(beta) - {x}, B.Dom) && forall b: T :: b in B.Dom ==> f[x := b] in setOfValmodels(beta, B)
    decreases x, beta, B
  {
    ghost var F := set f: Valuation<T> | f in allMaps(freeVar(beta) - {x}, B.Dom) && forall b: T :: b in B.Dom ==> f[x := b] in setOfValmodels(beta, B);
    forall f: Valuation<T> | f in F
      ensures f in setOfValmodels(Forall(x, beta), B)
    {
      assert f in allMaps(freeVar(Forall(x, beta)), B.Dom);
      assert f.Values <= B.Dom;
      assert models(B, f, Forall(x, beta));
    }
    forall f: Valuation<T> | f in setOfValmodels(Forall(x, beta), B)
      ensures f in F
    {
      assert f in allMaps(freeVar(beta) - {x}, B.Dom) && models(B, f, Forall(x, beta));
      forall b: T | b in B.Dom
        ensures f[x := b] in setOfValmodels(beta, B)
      {
        assert f[x := b].Values <= B.Dom;
        assert models(B, f[x := b], beta);
        ExtMap_Lemma(f, x, b, freeVar(beta), B.Dom);
        assert f[x := b] in allMaps(freeVar(beta), B.Dom);
        assert f[x := b] in setOfValmodels(beta, B);
      }
      assert forall b: T :: b in B.Dom ==> f[x := b] in setOfValmodels(beta, B);
      assert f in F;
    }
  }

  lemma /*{:_induction beta}*/ setofValmodelsExists_Lemma<T>(x: Name, beta: Formula, B: Structure<T>)
    requires wfStructure(B) && wfFormula(B.Sig, Exists(x, beta))
    requires x in freeVar(beta)
    ensures setOfValmodels(Exists(x, beta), B) == set h: map<Name, T> {:trigger h in setOfValmodels(beta, B)} | h in setOfValmodels(beta, B) :: projectVal(h, freeVar(beta) - {x})
    decreases x, beta, B
  {
    ghost var F := set h: map<Name, T> {:trigger h in setOfValmodels(beta, B)} | h in setOfValmodels(beta, B) :: projectVal(h, freeVar(beta) - {x});
    forall f: Valuation<T> | f in setOfValmodels(Exists(x, beta), B)
      ensures f in F
    {
      ghost var v :| v in B.Dom && models(B, f[x := v], beta);
      ghost var h := f[x := v];
      ExtMap_Lemma(f, x, v, freeVar(beta), B.Dom);
      assert h in allMaps(freeVar(beta), B.Dom);
      assert h in setOfValmodels(beta, B);
      assert f == projectVal(h, freeVar(beta) - {x});
      assert f in F;
    }
    forall f: Valuation<T> | f in F
      ensures f in setOfValmodels(Exists(x, beta), B)
    {
      ghost var h :| h in setOfValmodels(beta, B) && f == projectVal(h, freeVar(beta) - {x});
      assert h in allMaps(freeVar(beta), B.Dom);
      ProyectMap_Lemma(h, freeVar(beta) - {x}, freeVar(beta), B.Dom);
      assert f in allMaps(freeVar(beta) - {x}, B.Dom);
      Existsmodels_Lemma(B, h, x, beta);
      assert models(B, f, Exists(x, beta));
      assert f in setOfValmodels(Exists(x, beta), B);
    }
  }

  lemma /*{:_induction i, phi, B}*/ canonical_judgement_Lemma<T>(i: seq<int>, phi: Formula, B: Structure<T>)
    requires wfQCSP_Instance(phi, B)
    requires i in setOfIndex(phi)
    ensures canonical_judgement(i, phi, B).i == i
    ensures canonical_judgement(i, phi, B).V == freeVar(FoI(i, phi, B.Sig))
    ensures canonical_judgement(i, phi, B).F == setOfValmodels(FoI(i, phi, B.Sig), B)
    ensures is_derivable(canonical_judgement(i, phi, B), phi, B)
    decreases FoI(i, phi, B.Sig)
  {
    ghost var phii := FoI(i, phi, B.Sig);
    if !phii.Atom? {
      indexSubformula_Lemma(i, phi, B.Sig);
    }
    ghost var cj := canonical_judgement(i, phi, B);
    match phii
    case Atom(R, par) =>
      assert is_derivable(cj, phi, B);
    case And(phi0, phi1) =>
      var j0' := canonical_judgement(i + [0], phi, B);
      var j1' := canonical_judgement(i + [1], phi, B);
      var j0 := J(i, j0'.V, j0'.F);
      var j1 := J(i, j1'.V, j1'.F);
      canonical_judgement_Lemma(i + [0], phi, B);
      canonical_judgement_Lemma(i + [1], phi, B);
      setofValmodelsAnd_Lemma(phi0, phi1, B);
      assert cj.F == setOfValmodels(phii, B);
      assert is_derivable(j0, phi, B) && is_derivable(j1, phi, B);
      assert is_derivable(cj, phi, B);
    case Forall(x, beta) =>
      var j0 := canonical_judgement(i + [0], phi, B);
      canonical_judgement_Lemma(i + [0], phi, B);
      if x !in j0.V {
        forall f: Valuation<T> | f.Values <= B.Dom {
          NoFreeVarInForall_Lemma(B, f, x, beta);
        }
        assert cj.F == setOfValmodels(beta, B) == setOfValmodels(Forall(x, beta), B);
        assert is_derivable(cj, phi, B);
      } else {
        assert cj == dualProjection(x, j0, phi, B);
        setofValmodelsForall_Lemma(x, beta, B);
        assert cj.F == setOfValmodels(Forall(x, beta), B);
        assert is_derivable(cj, phi, B);
      }
    case Exists(x, beta) =>
      var j0 := canonical_judgement(i + [0], phi, B);
      canonical_judgement_Lemma(i + [0], phi, B);
      if x !in j0.V {
        forall f: Valuation<T> | f.Values <= B.Dom {
          NoFreeVarInExists_Lemma(B, f, x, beta);
        }
        assert cj.F == setOfValmodels(beta, B);
        assert is_derivable(cj, phi, B);
      } else {
        var jp := projection(j0, j0.V - {x}, phi, B);
        assert cj == J(i, jp.V, jp.F);
        assert cj.F == set h | h in setOfValmodels(beta, B) :: projectVal(h, j0.V - {x});
        setofValmodelsExists_Lemma(x, beta, B);
        assert cj.F == setOfValmodels(Exists(x, beta), B);
        assert is_derivable(j0, phi, B);
        assert is_derivable(cj, phi, B);
      }
  }

  lemma /*{:_induction phi, B}*/ completeness_Theorem<T>(phi: Formula, B: Structure<T>)
    requires wfQCSP_Instance(phi, B)
    requires !models(B, map[], phi)
    ensures is_derivable(J([], {}, {}), phi, B)
    decreases phi, B
  {
    canonical_judgement_Lemma([], phi, B);
    forall f: Valuation<T> | true
      ensures f !in canonical_judgement([], phi, B).F
    {
      calc ==> {
        f in canonical_judgement([], phi, B).F;
        f.Values <= B.Dom &&
        f.Keys == {} &&
        models(B, f, phi);
        f.Values <= B.Dom &&
        f == map[] &&
        models(B, f, phi);
        f.Values <= B.Dom &&
        models(B, map[], phi);
        false;
      }
    }
    assert canonical_judgement([], phi, B).F == {};
    assert is_derivable(J([], {}, {}), phi, B);
  }
}

module Proof_System {

  import opened Utils = Utils

  import opened QCSP_for_PS = QCSP_Instance`{Defs_for_Proof_System}

  export Lemma_and_Defs
    provides Utils, QCSP_for_PS, indexSubformula_Lemma
    reveals Index, Judgement, wfJudgement, setOfIndex, FoI, is_projection, is_join, is_dualProjection, is_upwardFlow, is_derivable

  type Index = seq<int>

  datatype Judgement<T(==)> = J(i: Index, V: set<Name>, F: set<Valuation<T>>)

  function setOfIndex(phi: Formula): set<Index>
    decreases phi
  {
    match phi
    case Atom(R, par) =>
      {[]}
    case And(phi, psi) =>
      {[]} + (set s | s in setOfIndex(phi) :: [0] + s) + set s | s in setOfIndex(psi) :: [1] + s
    case Forall(x, phi) =>
      {[]} + set s | s in setOfIndex(phi) :: [0] + s
    case Exists(x, phi) =>
      {[]} + set s | s in setOfIndex(phi) :: [0] + s
  }

  function method FoI(s: Index, phi: Formula, Sig: Signature): Formula
    requires s in setOfIndex(phi)
    ensures subformula(FoI(s, phi, Sig), phi)
    ensures FoI(s, phi, Sig) < phi <==> s != []
    ensures wfFormula(Sig, phi) ==> wfFormula(Sig, FoI(s, phi, Sig))
    decreases s, phi, Sig
  {
    if s == [] then
      phi
    else
      match phi case And(phi, psi) => (if s[0] == 0 then FoI(s[1..], phi, Sig) else FoI(s[1..], psi, Sig)) case Forall(x, phi) => FoI(s[1..], phi, Sig) case Exists(x, phi) => FoI(s[1..], phi, Sig)
  }

  lemma /*{:_induction alpha, phi, s, Sig}*/ Index_Lemma(alpha: Formula, phi: Formula, s: Index, Sig: Signature)
    requires s in setOfIndex(phi) && FoI(s, phi, Sig) == alpha
    requires alpha != phi
    ensures phi.And? ==> s != [] && ((subformula(alpha, phi.0) && s[0] == 0 && s[1..] in setOfIndex(phi.0) && FoI(s[1..], phi.0, Sig) == alpha) || (subformula(alpha, phi.1) && s[0] == 1 && s[1..] in setOfIndex(phi.1) && FoI(s[1..], phi.1, Sig) == alpha))
    ensures phi.Forall? ==> s != [] && subformula(alpha, phi.Body) && s[0] == 0 && s[1..] in setOfIndex(phi.Body) && FoI(s[1..], phi.Body, Sig) == alpha
    ensures phi.Exists? ==> s != [] && subformula(alpha, phi.Body) && s[0] == 0 && s[1..] in setOfIndex(phi.Body) && FoI(s[1..], phi.Body, Sig) == alpha
    decreases alpha, phi, s, Sig
  {
  }

  lemma /*{:_induction phi, alpha, r1, r2, Sig}*/ IndexSum_Lemma(phi: Formula, alpha: Formula, beta: Formula, r1: Index, r2: Index, Sig: Signature)
    requires r1 in setOfIndex(phi) && FoI(r1, phi, Sig) == alpha
    requires r2 in setOfIndex(alpha) && FoI(r2, alpha, Sig) == beta
    ensures r1 + r2 in setOfIndex(phi) && FoI(r1 + r2, phi, Sig) == beta
    decreases r1
  {
  }

  lemma /*{:_induction s, phi, Sig}*/ indexSubformula_Lemma(s: Index, phi: Formula, Sig: Signature)
    requires s in setOfIndex(phi) && !FoI(s, phi, Sig).Atom?
    ensures FoI(s, phi, Sig).And? ==> s + [0] in setOfIndex(phi) && FoI(s, phi, Sig).0 == FoI(s + [0], phi, Sig) && s + [1] in setOfIndex(phi) && FoI(s, phi, Sig).1 == FoI(s + [1], phi, Sig)
    ensures FoI(s, phi, Sig).Forall? ==> s + [0] in setOfIndex(phi) && FoI(s, phi, Sig).Body == FoI(s + [0], phi, Sig) && freeVar(FoI(s, phi, Sig)) == freeVar(FoI(s + [0], phi, Sig)) - {FoI(s, phi, Sig).x}
    ensures FoI(s, phi, Sig).Exists? ==> s + [0] in setOfIndex(phi) && FoI(s, phi, Sig).Body == FoI(s + [0], phi, Sig) && freeVar(FoI(s, phi, Sig)) == freeVar(FoI(s + [0], phi, Sig)) - {FoI(s, phi, Sig).x}
    decreases s, phi, Sig
  {
  }

  lemma /*{:_induction s, phi, Sig}*/ setOfIndexSuffix_Lemma(s: Index, phi: Formula, Sig: Signature)
    requires s in setOfIndex(phi)
    ensures !FoI(s, phi, Sig).Atom? ==> s + [0] in setOfIndex(phi)
    ensures FoI(s, phi, Sig).And? ==> s + [1] in setOfIndex(phi)
    decreases s, phi, Sig
  {
  }

  predicate wfJudgement<T>(j: Judgement<T>, phi: Formula, B: Structure<T>)
    decreases j, phi, B
  {
    wfQCSP_Instance(phi, B) &&
    j.i in setOfIndex(phi) &&
    j.V <= freeVar(FoI(j.i, phi, B.Sig)) &&
    forall f: map<Name, T> :: 
      f in j.F ==>
        j.V == f.Keys
  }

  predicate is_projection<T>(j1: Judgement<T>, j2: Judgement<T>, phi: Formula, B: Structure<T>)
    requires wfJudgement(j1, phi, B) && wfJudgement(j2, phi, B)
    decreases j1, j2, phi, B
  {
    j1.i == j2.i &&
    j1.V <= j2.V &&
    j1.F == set f: map<Name, T> {:trigger projectVal(f, j1.V)} {:trigger f in j2.F} | f in j2.F :: projectVal(f, j1.V)
  }

  predicate is_join<T>(j: Judgement<T>, j1: Judgement<T>, j2: Judgement<T>, phi: Formula, B: Structure<T>)
    requires wfJudgement(j, phi, B) && wfJudgement(j1, phi, B) && wfJudgement(j2, phi, B)
    decreases j, j1, j2, phi, B
  {
    j.i == j1.i == j2.i &&
    j.V == j1.V + j2.V &&
    j.F == set f: Valuation<T> {:trigger projectVal(f, j2.V)} {:trigger projectVal(f, j1.V)} | f in allMaps(j1.V + j2.V, B.Dom) && projectVal(f, j1.V) in j1.F && projectVal(f, j2.V) in j2.F
  }

  predicate is_dualProjection<T>(j1: Judgement<T>, v: Name, j2: Judgement<T>, phi: Formula, B: Structure<T>)
    requires wfJudgement(j1, phi, B) && wfJudgement(j2, phi, B)
    decreases j1, v, j2, phi, B
  {
    j2.i == j1.i + [0] &&
    j1.V == j2.V - {v} &&
    v in j2.V &&
    j1.F == set h: Valuation<T> {:trigger h in allMaps(j1.V, B.Dom)} | h in allMaps(j1.V, B.Dom) && forall b: T :: b in B.Dom ==> h[v := b] in j2.F
  }

  predicate is_upwardFlow<T>(j1: Judgement<T>, j2: Judgement<T>, phi: Formula, B: Structure<T>)
    requires wfJudgement(j1, phi, B) && wfJudgement(j2, phi, B)
    decreases j1, j2, phi, B
  {
    j2.V == j1.V &&
    j2.F == j1.F &&
    ((FoI(j1.i, phi, B.Sig).And? && (j2.i == j1.i + [0] || j2.i == j1.i + [1])) || ((FoI(j1.i, phi, B.Sig).Forall? || FoI(j1.i, phi, B.Sig).Exists?) && j2.i == j1.i + [0]))
  }

  inductive predicate is_derivable<T(!new)>(j: Judgement<T>, phi: Formula, B: Structure<T>)
    requires wfQCSP_Instance(phi, B) && wfJudgement(j, phi, B)
  {
    var phii: Formula := FoI(j.i, phi, B.Sig);
    (phii.Atom? && j.V == setOf(phii.par) && j.F == set f: Valuation<T> | f in allMaps(j.V, B.Dom) && HOmap(f, phii.par) in B.I[phii.rel]) || (exists j': Judgement<T> :: wfJudgement(j', phi, B) && is_projection(j, j', phi, B) && is_derivable(j', phi, B)) || (phii.And? && exists j0: Judgement<T>, j1: Judgement<T> :: wfJudgement(j0, phi, B) && wfJudgement(j1, phi, B) && j0.i == j1.i && is_join(j, j0, j1, phi, B) && is_derivable(j0, phi, B) && is_derivable(j1, phi, B)) || (phii.Forall? && exists j': Judgement<T> :: wfJudgement(j', phi, B) && phii == Forall(phii.x, FoI(j'.i, phi, B.Sig)) && is_dualProjection(j, phii.x, j', phi, B) && is_derivable(j', phi, B)) || (!phii.Atom? && exists j': Judgement<T> :: wfJudgement(j', phi, B) && is_upwardFlow(j, j', phi, B) && is_derivable(j', phi, B))
  }
  /*** (note, what is printed here does not show substitutions of calls to prefix predicates)
  predicate is_derivable#<T(!new)>[_k: ORDINAL](j: Judgement<T>, phi: Formula, B: Structure<T>)
    requires wfQCSP_Instance(phi, B) && wfJudgement(j, phi, B)
    decreases _k
  {
    var phii: Formula := FoI(j.i, phi, B.Sig);
    (phii.Atom? && j.V == setOf(phii.par) && j.F == set f: Valuation<T> | f in allMaps(j.V, B.Dom) && HOmap(f, phii.par) in B.I[phii.rel]) || (exists j': Judgement<T> :: wfJudgement(j', phi, B) && is_projection(j, j', phi, B) && is_derivable(j', phi, B)) || (phii.And? && exists j0: Judgement<T>, j1: Judgement<T> :: wfJudgement(j0, phi, B) && wfJudgement(j1, phi, B) && j0.i == j1.i && is_join(j, j0, j1, phi, B) && is_derivable(j0, phi, B) && is_derivable(j1, phi, B)) || (phii.Forall? && exists j': Judgement<T> :: wfJudgement(j', phi, B) && phii == Forall(phii.x, FoI(j'.i, phi, B.Sig)) && is_dualProjection(j, phii.x, j', phi, B) && is_derivable(j', phi, B)) || (!phii.Atom? && exists j': Judgement<T> :: wfJudgement(j', phi, B) && is_upwardFlow(j, j', phi, B) && is_derivable(j', phi, B))
  }
  ***/
}

module QCSP_Instance {

  import opened Utils = Utils

  export Lemmas_for_Existential_Closure
    reveals Name, Valuation, Structure, Signature, Formula, extVal, wfStructure, wfFormula, models, freeVar, HOmap, sentence, projectVal
    provides Utils, extValDomRange_Lemma, extValOrder_Lemma, NoFreeVarInExists_Lemma, Exists_Commutes_Lemma


  export Defs_for_Proof_System
    provides Utils
    reveals Name, Formula, Signature, Valuation, Structure, subformula, wfFormula, freeVar, projectVal, wfStructure, sentence, wfQCSP_Instance, HOmap


  export Lemmas_for_PS_Soundness
    provides Utils, extValOrder_Lemma, Exists_Commutes_Lemma, NoFreeVarInExists_Lemma, extValCommute_Lemma, projectOfExtVal_Lemma, extValDomRange_Lemma, extValallMaps_Lemma
    reveals Name, Valuation, Structure, Signature, Formula, extVal, projectVal, wfStructure, wfFormula, models, freeVar, HOmap, sentence, wfQCSP_Instance


  export Lemmas_for_PS_Completeness
    provides Utils, modelsAnd_Lemma, Existsmodels_Lemma, NoFreeVarInExists_Lemma, NoFreeVarInForall_Lemma, extValCommute_Lemma
    reveals Name, Valuation, extVal, Structure, Signature, Formula, projectVal, HOmap, wfFormula, wfStructure, freeVar, sentence, wfQCSP_Instance, models

  type Name = string

  type Signature = map<Name, int>

  datatype Structure<T> = Structure(Sig: Signature, Dom: set<T>, I: map<Name, set<seq<T>>>)

  datatype Formula = Atom(rel: Name, par: seq<Name>) | And(0: Formula, 1: Formula) | Forall(x: Name, Body: Formula) | Exists(x: Name, Body: Formula)

  type Valuation<T> = map<Name, T>

  predicate wfStructure<T>(B: Structure<T>)
    decreases B
  {
    B.Dom != {} &&
    forall r: seq<char> :: 
      r in B.Sig.Keys ==>
        r in B.I &&
        forall t: seq<T> :: 
          t in B.I[r] ==>
            |t| == B.Sig[r]
  }

  predicate wfFormula(S: Signature, phi: Formula)
    decreases S, phi
  {
    match phi
    case Atom(R, par) =>
      R in S.Keys &&
      |par| == S[R]
    case And(phi0, phi1) =>
      wfFormula(S, phi0) &&
      wfFormula(S, phi1)
    case Forall(x, alpha) =>
      wfFormula(S, alpha)
    case Exists(x, alpha) =>
      wfFormula(S, alpha)
  }

  predicate subformula(beta: Formula, phi: Formula)
    decreases beta, phi
  {
    beta == phi || (beta < phi && ((phi.And? && (subformula(beta, phi.0) || subformula(beta, phi.1))) || (phi.Forall? && subformula(beta, phi.Body)) || (phi.Exists? && subformula(beta, phi.Body))))
  }

  function freeVar(phi: Formula): set<Name>
    decreases phi
  {
    match phi
    case Atom(R, par) =>
      setOf(par)
    case And(ph1, phi1) =>
      freeVar(ph1) + freeVar(phi1)
    case Forall(x, phi) =>
      freeVar(phi) - {x}
    case Exists(x, phi) =>
      freeVar(phi) - {x}
  }

  predicate sentence(phi: Formula)
    decreases phi
  {
    freeVar(phi) == {}
  }

  predicate wfQCSP_Instance(phi: Formula, B: Structure)
    decreases phi, B
  {
    wfStructure(B) &&
    wfFormula(B.Sig, phi) &&
    sentence(phi)
  }

  function method HOmap<U, V>(f: map<U, V>, xs: seq<U>): seq<V>
    requires setOf(xs) <= f.Keys
    ensures |HOmap(f, xs)| == |xs|
    decreases f, xs
  {
    if xs == [] then
      []
    else
      [f[xs[0]]] + HOmap(f, xs[1..])
  }

  function method projectVal<T>(f: Valuation<T>, U: set<Name>): Valuation<T>
    requires U <= f.Keys
    ensures projectVal(f, U).Keys == U
    decreases f, U
  {
    map s: seq<char> {:trigger f[s]} {:trigger s in U} | s in U :: f[s]
  }

  function extVal<T>(f: Valuation<T>, W: seq<Name>, S: seq<T>): Valuation<T>
    requires |W| == |S|
    requires noDups(W)
    decreases |W|
  {
    if W == [] then
      f
    else
      extVal(f[W[0] := S[0]], W[1..], S[1..])
  }

  lemma /*{:_induction f, W, S}*/ extValDomRange_Lemma<T>(f: Valuation<T>, W: seq<Name>, S: seq<T>)
    requires |W| == |S|
    requires noDups(W)
    ensures extVal(f, W, S).Keys == setOf(W) + f.Keys
    ensures extVal(f, W, S).Values <= f.Values + setOf(S)
    decreases W
  {
  }

  lemma /*{:_induction f, W, S}*/ extValallMaps_Lemma<T>(f: Valuation<T>, W: seq<Name>, S: seq<T>, B: Structure<T>)
    requires |W| == |S|
    requires noDups(W) && setOf(W) !! f.Keys && setOf(S) <= B.Dom
    requires f.Values <= B.Dom
    ensures extVal(f, W, S) in allMaps(extVal(f, W, S).Keys, B.Dom)
    decreases f, W, S, B
  {
  }

  lemma /*{:_induction f, xs}*/ HOmapProject_Lemma<T>(f: Valuation<T>, U: set<Name>, xs: seq<Name>)
    requires setOf(xs) <= U <= f.Keys
    ensures HOmap(projectVal(f, U), xs) == HOmap(f, xs)
    decreases f, U, xs
  {
  }

  lemma varNotInProjectVal_Lemma<T>(f: Valuation<T>, x: Name, a: T, U: set<Name>)
    requires x !in U
    requires U <= f.Keys
    ensures projectVal(f[x := a], U) == projectVal(f, U)
    decreases f, x, U
  {
  }

  lemma /*{:_induction f, U, Z}*/ ExtVal_Lemma<T>(f: Valuation<T>, U: seq<Name>, Z: seq<T>, x: Name)
    requires |U| == |Z| && noDups(U) && setOf(U) !! f.Keys
    requires x in f.Keys
    ensures x in extVal(f, U, Z).Keys
    ensures extVal(f, U, Z)[x] == f[x]
    decreases U
  {
  }

  lemma ExtProjectVal_Lemma<T>(B: Structure<T>, f: Valuation<T>, x: Name, U: set<Name>, v: T)
    requires v in B.Dom && x in U && U <= f.Keys && f[x] == v
    ensures projectVal(f, U) == projectVal(f, U - {x})[x := v]
    decreases B, f, x, U
  {
  }

  lemma /*{:_induction f, U, Z}*/ projectOfExtVal_Lemma<T>(f: Valuation<T>, U: seq<Name>, Z: seq<T>)
    requires |U| == |Z| && noDups(U) && setOf(U) !! f.Keys
    ensures f.Keys <= extVal(f, U, Z).Keys
    ensures projectVal(extVal(f, U, Z), f.Keys) == f
    decreases f, U, Z
  {
  }

  lemma /*{:_induction f, U, Z}*/ extValCommute_Lemma<T>(x: Name, v: T, f: Valuation<T>, U: seq<Name>, Z: seq<T>)
    requires |U| == |Z| && noDups(U)
    requires x !in U
    ensures extVal(f, U, Z)[x := v] == extVal(f[x := v], U, Z)
    decreases |U|
  {
  }

  lemma /*{:_induction U, S, f}*/ extValOrder_Lemma<T>(k: int, U: seq<Name>, S: seq<T>, f: Valuation<T>)
    requires 0 <= k < |U| == |S|
    requires noDups(U)
    ensures extVal(f, U, S) == extVal(f[U[k] := S[k]], U[..k] + U[k + 1..], S[..k] + S[k + 1..])
    decreases k, U, S, f
  {
  }

  lemma /*{:_induction f, W, W', V, V'}*/ extValSum_Lemma<T>(f: Valuation<T>, W: seq<Name>, W': seq<Name>, V: seq<T>, V': seq<T>)
    requires |W| == |V| && |W'| == |V'| && noDups(W) && noDups(W') && noDups(W + W')
    ensures extVal(extVal(f, W, V), W', V') == extVal(f, W + W', V + V')
    decreases |W|
  {
  }

  predicate models<T>(B: Structure<T>, f: Valuation<T>, phi: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, phi) && f.Values <= B.Dom
    decreases phi
  {
    freeVar(phi) <= f.Keys &&
    match phi case Atom(R, par) => HOmap(f, par) in B.I[R] case And(phi0, phi1) => models(B, f, phi0) && models(B, f, phi1) case Forall(x, alpha) => (forall v :: v in B.Dom ==> models(B, f[x := v], alpha)) case Exists(x, alpha) => exists v :: v in B.Dom && models(B, f[x := v], alpha)
  }

  lemma /*{:_induction B, f, phi}*/ safeProjection_Lemma<T>(B: Structure<T>, f: Valuation<T>, U: set<Name>, phi: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, phi) && f.Values <= B.Dom
    requires freeVar(phi) <= U <= f.Keys
    ensures models(B, f, phi) <==> models(B, projectVal(f, U), phi)
    decreases phi
  {
  }

  lemma /*{:_induction B, f, g, phi}*/ freeVarsDep_Lemma<T>(B: Structure<T>, f: Valuation<T>, g: Valuation<T>, phi: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, phi) && f.Values <= B.Dom && g.Values <= B.Dom
    requires freeVar(phi) <= f.Keys && freeVar(phi) <= g.Keys
    requires projectVal(f, freeVar(phi)) == projectVal(g, freeVar(phi))
    requires models(B, f, phi)
    ensures models(B, g, phi)
    decreases B, f, g, phi
  {
  }

  lemma /*{:_induction B, f, phi}*/ modelsAnd_Lemma<T>(B: Structure<T>, f: Valuation<T>, phi: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, phi) && f.Values <= B.Dom
    requires phi.And?
    ensures models(B, f, phi) <==> freeVar(phi.0) <= f.Keys && models(B, projectVal(f, freeVar(phi.0)), phi.0) && freeVar(phi.1) <= f.Keys && models(B, projectVal(f, freeVar(phi.1)), phi.1)
    decreases B, f, phi
  {
  }

  lemma /*{:_induction B, f, beta}*/ NoFreeVar_Lemma<T>(B: Structure<T>, f: Valuation<T>, x: Name, a: T, beta: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, beta) && f.Values <= B.Dom
    requires x !in freeVar(beta)
    requires a in B.Dom
    ensures models(B, f[x := a], beta) <==> models(B, f, beta)
    decreases B, f, x, beta
  {
  }

  lemma /*{:_induction B, f, beta}*/ NoFreeVarInForall_Lemma<T>(B: Structure<T>, f: Valuation<T>, x: Name, beta: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, beta) && f.Values <= B.Dom
    requires x !in freeVar(beta)
    ensures models(B, f, beta) <==> models(B, f, Forall(x, beta))
    decreases B, f, x, beta
  {
  }

  lemma /*{:_induction B, f, beta}*/ NoFreeVarInExists_Lemma<T>(B: Structure<T>, f: Valuation<T>, x: Name, beta: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, beta) && f.Values <= B.Dom
    requires x !in freeVar(beta)
    ensures models(B, f, beta) <==> models(B, f, Exists(x, beta))
    decreases B, f, x, beta
  {
  }

  lemma /*{:_induction B, f, phi}*/ weakExists_Lemma<T>(B: Structure<T>, f: Valuation<T>, x: Name, phi: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, phi) && f.Values <= B.Dom
    requires models(B, f, phi)
    ensures models(B, f, Exists(x, phi))
    decreases B, f, x, phi
  {
  }

  lemma /*{:_induction B, f, beta}*/ Existsmodels_Lemma<T>(B: Structure<T>, f: Valuation<T>, x: Name, beta: Formula)
    requires wfStructure(B) && wfFormula(B.Sig, Exists(x, beta)) && f.Values <= B.Dom
    requires x in freeVar(beta) && models(B, f, beta)
    ensures models(B, projectVal(f, freeVar(beta) - {x}), Exists(x, beta))
    decreases B, f, x, beta
  {
  }

  lemma /*{:_induction alpha, f, B}*/ Exists_Commutes_Lemma<T>(x: Name, y: Name, alpha: Formula, f: Valuation<T>, B: Structure<T>)
    requires wfStructure(B) && wfFormula(B.Sig, alpha) && f.Values <= B.Dom
    requires models(B, f, Exists(x, Exists(y, alpha)))
    ensures models(B, f, Exists(y, Exists(x, alpha)))
    decreases x, y, alpha, f, B
  {
  }
}

module Utils {

  export 
    reveals setOf, noDups, allMaps, choose
    provides allMaps_Correct_Lemma, ExtMap_Lemma, ProyectMap_Lemma

  function setOf<T>(s: seq<T>): set<T>
    decreases s
  {
    set x: T {:trigger x in s} | x in s
  }

  predicate noDups<T(==)>(U: seq<T>)
    decreases U
  {
    forall i: int, j: int :: 
      0 <= i < j < |U| ==>
        U[i] != U[j]
  }

  function allMaps<A, B>(keys: set<A>, values: set<B>): set<map<A, B>>
    ensures forall m: map<A, B> :: m in allMaps(keys, values) ==> m.Keys == keys && m.Values <= values
    decreases keys, values
  {
    if keys == {} then
      {map[]}
    else
      var k: A := choose(keys); var M: set<map<A, B>> := allMaps(keys - {k}, values); set m: map<A, B>, v: B | m in M && v in values :: m[k := v]
  }

  function choose<A>(s: set<A>): A
    requires s != {}
    decreases s
  {
    var a: A :| a in s;
    a
  }

  lemma /*{:_induction values}*/ allMaps_Correct_Lemma<A, B>(p: map<A, B>, values: set<B>)
    requires p.Values <= values
    ensures p in allMaps(p.Keys, values)
    decreases p, values
  {
  }

  lemma /*{:_induction keys, values}*/ ExtMap_Lemma<A, B>(p: map<A, B>, x: A, v: B, keys: set<A>, values: set<B>)
    requires p in allMaps(keys - {x}, values)
    requires x in keys && v in values
    ensures p[x := v] in allMaps(keys, values)
    decreases p, keys, values
  {
  }

  lemma /*{:_induction subkeys, keys, values}*/ ProyectMap_Lemma<A, B>(p: map<A, B>, subkeys: set<A>, keys: set<A>, values: set<B>)
    requires p in allMaps(keys, values)
    requires subkeys <= keys
    ensures (map s: A {:trigger p[s]} {:trigger s in subkeys} | s in subkeys :: p[s]) in allMaps(subkeys, values)
    decreases p, subkeys, keys, values
  {
  }
}
")]

#if ISDAFNYRUNTIMELIB
using System; // for Func
using System.Numerics;
#endif

namespace DafnyAssembly {
  [AttributeUsage(AttributeTargets.Assembly)]
  public class DafnySourceAttribute : Attribute {
    public readonly string dafnySourceText;
    public DafnySourceAttribute(string txt) { dafnySourceText = txt; }
  }
}

namespace Dafny
{
  using System.Collections.Generic;
  // set this option if you want to use System.Collections.Immutable and if you know what you're doing.
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
  using System.Collections.Immutable;
  using System.Linq;
#endif

  public class Set<T>
  {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
    readonly ImmutableHashSet<T> setImpl;
    readonly bool containsNull;
    Set(ImmutableHashSet<T> d, bool containsNull) {
      this.setImpl = d;
      this.containsNull = containsNull;
    }
    public static readonly Set<T> Empty = new Set<T>(ImmutableHashSet<T>.Empty, false);
    public static Set<T> FromElements(params T[] values) {
      return FromCollection(values);
    }
    public static Set<T> FromCollection(IEnumerable<T> values) {
      var d = ImmutableHashSet<T>.Empty.ToBuilder();
      var containsNull = false;
      foreach (T t in values) {
        if (t == null) {
          containsNull = true;
        } else {
          d.Add(t);
        }
      }
      return new Set<T>(d.ToImmutable(), containsNull);
    }
    public static Set<T> FromCollectionPlusOne(IEnumerable<T> values, T oneMoreValue) {
      var d = ImmutableHashSet<T>.Empty.ToBuilder();
      var containsNull = false;
      if (oneMoreValue == null) {
        containsNull = true;
      } else {
        d.Add(oneMoreValue);
      }
      foreach (T t in values) {
        if (t == null) {
          containsNull = true;
        } else {
          d.Add(t);
        }
      }
      return new Set<T>(d.ToImmutable(), containsNull);
    }
    public int Length {
      get { return this.setImpl.Count + (containsNull ? 1 : 0); }
    }
    public long LongLength {
      get { return this.setImpl.Count + (containsNull ? 1 : 0); }
    }
    public IEnumerable<T> Elements {
      get {
        if (containsNull) {
          yield return default(T);
        }
        foreach (var t in this.setImpl) {
          yield return t;
        }
      }
    }
#else
    readonly HashSet<T> setImpl;
    Set(HashSet<T> s) {
      this.setImpl = s;
    }
    public static readonly Set<T> Empty = new Set<T>(new HashSet<T>());
    public static Set<T> FromElements(params T[] values) {
      return FromCollection(values);
    }
    public static Set<T> FromCollection(IEnumerable<T> values) {
      var s = new HashSet<T>(values);
      return new Set<T>(s);
    }
    public static Set<T> FromCollectionPlusOne(IEnumerable<T> values, T oneMoreValue) {
      var s = new HashSet<T>(values);
      s.Add(oneMoreValue);
      return new Set<T>(s);
    }
    public int Length {
      get { return this.setImpl.Count; }
    }
    public long LongLength {
      get { return this.setImpl.Count; }
    }
    public IEnumerable<T> Elements {
      get {
        return this.setImpl;
      }
    }
#endif

    /// <summary>
    /// This is an inefficient iterator for producing all subsets of "this".
    /// </summary>
    public IEnumerable<Set<T>> AllSubsets {
      get {
        // Start by putting all set elements into a list, but don't include null
        var elmts = new List<T>();
        elmts.AddRange(this.setImpl);
        var n = elmts.Count;
        var which = new bool[n];
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
        var s = ImmutableHashSet<T>.Empty.ToBuilder();
#else
        var s = new HashSet<T>();
#endif
        while (true) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
          // yield both the subset without null and, if null is in the original set, the subset with null included
          var ihs = s.ToImmutable();
          yield return new Set<T>(ihs, false);
          if (containsNull) {
            yield return new Set<T>(ihs, true);
          }
#else
          yield return new Set<T>(new HashSet<T>(s));
#endif
          // "add 1" to "which", as if doing a carry chain.  For every digit changed, change the membership of the corresponding element in "s".
          int i = 0;
          for (; i < n && which[i]; i++) {
            which[i] = false;
            s.Remove(elmts[i]);
          }
          if (i == n) {
            // we have cycled through all the subsets
            break;
          }
          which[i] = true;
          s.Add(elmts[i]);
        }
      }
    }
    public bool Equals(Set<T> other) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      return containsNull == other.containsNull && this.setImpl.SetEquals(other.setImpl);
#else
      return this.setImpl.Count == other.setImpl.Count && IsSubsetOf(other);
#endif
    }
    public override bool Equals(object other) {
      return other is Set<T> && Equals((Set<T>)other);
    }
    public override int GetHashCode() {
      var hashCode = 1;
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      if (containsNull) {
        hashCode = hashCode * (Dafny.Helpers.GetHashCode(default(T)) + 3);
      }
#endif
      foreach (var t in this.setImpl) {
        hashCode = hashCode * (Dafny.Helpers.GetHashCode(t)+3);
      }
      return hashCode;
    }
    public override string ToString() {
      var s = "{";
      var sep = "";
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      if (containsNull) {
        s += sep + Dafny.Helpers.ToString(default(T));
        sep = ", ";
      }
#endif
      foreach (var t in this.setImpl) {
        s += sep + Dafny.Helpers.ToString(t);
        sep = ", ";
      }
      return s + "}";
    }
    public bool IsProperSubsetOf(Set<T> other) {
      return this.Length < other.Length && IsSubsetOf(other);
    }
    public bool IsSubsetOf(Set<T> other) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      if (this.containsNull && !other.containsNull) {
        return false;
      }
#endif
      if (other.setImpl.Count < this.setImpl.Count)
        return false;
      foreach (T t in this.setImpl) {
        if (!other.setImpl.Contains(t))
          return false;
      }
      return true;
    }
    public bool IsSupersetOf(Set<T> other) {
      return other.IsSubsetOf(this);
    }
    public bool IsProperSupersetOf(Set<T> other) {
      return other.IsProperSubsetOf(this);
    }
    public bool IsDisjointFrom(Set<T> other) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      if (this.containsNull && other.containsNull) {
        return false;
      }
      ImmutableHashSet<T> a, b;
#else
      HashSet<T> a, b;
#endif
      if (this.setImpl.Count < other.setImpl.Count) {
        a = this.setImpl; b = other.setImpl;
      } else {
        a = other.setImpl; b = this.setImpl;
      }
      foreach (T t in a) {
        if (b.Contains(t))
          return false;
      }
      return true;
    }
    public bool Contains<G>(G t) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      return t == null ? containsNull : t is T && this.setImpl.Contains((T)(object)t);
#else
      return (t == null || t is T) && this.setImpl.Contains((T)(object)t);
#endif
    }
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
    public Set<T> Union(Set<T> other) {
      return new Set<T>(this.setImpl.Union(other.setImpl), containsNull || other.containsNull);
    }
    public Set<T> Intersect(Set<T> other) {
      return new Set<T>(this.setImpl.Intersect(other.setImpl), containsNull && other.containsNull);
    }
    public Set<T> Difference(Set<T> other) {
        return new Set<T>(this.setImpl.Except(other.setImpl), containsNull && !other.containsNull);
    }
#else
    public Set<T> Union(Set<T> other) {
      if (this.setImpl.Count == 0)
        return other;
      else if (other.setImpl.Count == 0)
        return this;
      HashSet<T> a, b;
      if (this.setImpl.Count < other.setImpl.Count) {
        a = this.setImpl; b = other.setImpl;
      } else {
        a = other.setImpl; b = this.setImpl;
      }
      var r = new HashSet<T>();
      foreach (T t in b)
        r.Add(t);
      foreach (T t in a)
        r.Add(t);
      return new Set<T>(r);
    }
    public Set<T> Intersect(Set<T> other) {
      if (this.setImpl.Count == 0)
        return this;
      else if (other.setImpl.Count == 0)
        return other;
      HashSet<T> a, b;
      if (this.setImpl.Count < other.setImpl.Count) {
        a = this.setImpl; b = other.setImpl;
      } else {
        a = other.setImpl; b = this.setImpl;
      }
      var r = new HashSet<T>();
      foreach (T t in a) {
        if (b.Contains(t))
          r.Add(t);
      }
      return new Set<T>(r);
    }
    public Set<T> Difference(Set<T> other) {
      if (this.setImpl.Count == 0)
        return this;
      else if (other.setImpl.Count == 0)
        return this;
      var r = new HashSet<T>();
      foreach (T t in this.setImpl) {
        if (!other.setImpl.Contains(t))
          r.Add(t);
      }
      return new Set<T>(r);
    }
#endif
  }

  public class MultiSet<T>
  {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
    readonly ImmutableDictionary<T, int> dict;
#else
    readonly Dictionary<T, int> dict;
#endif
    readonly BigInteger occurrencesOfNull;  // stupidly, a Dictionary in .NET cannot use "null" as a key
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
    MultiSet(ImmutableDictionary<T, int>.Builder d, BigInteger occurrencesOfNull) {
      dict = d.ToImmutable();
      this.occurrencesOfNull = occurrencesOfNull;
    }
    public static readonly MultiSet<T> Empty = new MultiSet<T>(ImmutableDictionary<T, int>.Empty.ToBuilder(), BigInteger.Zero);
#else
    MultiSet(Dictionary<T, int> d, BigInteger occurrencesOfNull) {
      this.dict = d;
      this.occurrencesOfNull = occurrencesOfNull;
    }
    public static MultiSet<T> Empty = new MultiSet<T>(new Dictionary<T, int>(0), BigInteger.Zero);
#endif
    public static MultiSet<T> FromElements(params T[] values) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var d = ImmutableDictionary<T, int>.Empty.ToBuilder();
#else
      var d = new Dictionary<T, int>(values.Length);
#endif
      var occurrencesOfNull = BigInteger.Zero;
      foreach (T t in values) {
        if (t == null) {
          occurrencesOfNull++;
        } else {
          var i = 0;
          if (!d.TryGetValue(t, out i)) {
            i = 0;
          }
          d[t] = i + 1;
        }
      }
      return new MultiSet<T>(d, occurrencesOfNull);
    }
    public static MultiSet<T> FromCollection(ICollection<T> values) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var d = ImmutableDictionary<T, int>.Empty.ToBuilder();
#else
      var d = new Dictionary<T, int>();
#endif
      var occurrencesOfNull = BigInteger.Zero;
      foreach (T t in values) {
        if (t == null) {
          occurrencesOfNull++;
        } else {
          var i = 0;
          if (!d.TryGetValue(t, out i)) {
            i = 0;
          }
          d[t] = i + 1;
        }
      }
      return new MultiSet<T>(d, occurrencesOfNull);
    }
    public static MultiSet<T> FromSeq(Sequence<T> values) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var d = ImmutableDictionary<T, int>.Empty.ToBuilder();
#else
      var d = new Dictionary<T, int>();
#endif
      var occurrencesOfNull = BigInteger.Zero;
      foreach (T t in values.Elements) {
        if (t == null) {
          occurrencesOfNull++;
        } else {
          var i = 0;
          if (!d.TryGetValue(t, out i)) {
            i = 0;
          }
          d[t] = i + 1;
        }
      }
      return new MultiSet<T>(d, occurrencesOfNull);
    }
    public static MultiSet<T> FromSet(Set<T> values) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var d = ImmutableDictionary<T, int>.Empty.ToBuilder();
#else
      var d = new Dictionary<T, int>();
#endif
      var containsNull = false;
      foreach (T t in values.Elements) {
        if (t == null) {
          containsNull = true;
        } else {
          d[t] = 1;
        }
      }
      return new MultiSet<T>(d, containsNull ? BigInteger.One : BigInteger.Zero);
    }

    public bool Equals(MultiSet<T> other) {
      return other.IsSubsetOf(this) && this.IsSubsetOf(other);
    }
    public override bool Equals(object other) {
      return other is MultiSet<T> && Equals((MultiSet<T>)other);
    }
    public override int GetHashCode() {
      var hashCode = 1;
      if (occurrencesOfNull > 0) {
        var key = Dafny.Helpers.GetHashCode(default(T));
        key = (key << 3) | (key >> 29) ^ occurrencesOfNull.GetHashCode();
        hashCode = hashCode * (key + 3);
      }
      foreach (var kv in dict) {
        var key = Dafny.Helpers.GetHashCode(kv.Key);
        key = (key << 3) | (key >> 29) ^ kv.Value.GetHashCode();
        hashCode = hashCode * (key + 3);
      }
      return hashCode;
    }
    public override string ToString() {
      var s = "multiset{";
      var sep = "";
      for (var i = BigInteger.Zero; i < occurrencesOfNull; i++) {
        s += sep + Dafny.Helpers.ToString(default(T));
        sep = ", ";
      }
      foreach (var kv in dict) {
        var t = Dafny.Helpers.ToString(kv.Key);
        for (int i = 0; i < kv.Value; i++) {
          s += sep + t;
          sep = ", ";
        }
      }
      return s + "}";
    }
    public bool IsProperSubsetOf(MultiSet<T> other) {
      return !Equals(other) && IsSubsetOf(other);
    }
    public bool IsSubsetOf(MultiSet<T> other) {
      if (other.occurrencesOfNull < this.occurrencesOfNull) {
        return false;
      }
      foreach (T t in dict.Keys) {
        if (!other.dict.ContainsKey(t) || other.dict[t] < dict[t])
          return false;
      }
      return true;
    }
    public bool IsSupersetOf(MultiSet<T> other) {
      return other.IsSubsetOf(this);
    }
    public bool IsProperSupersetOf(MultiSet<T> other) {
      return other.IsProperSubsetOf(this);
    }
    public bool IsDisjointFrom(MultiSet<T> other) {
      if (occurrencesOfNull > 0 && other.occurrencesOfNull > 0) {
        return false;
      }
      foreach (T t in dict.Keys) {
        if (other.dict.ContainsKey(t))
          return false;
      }
      foreach (T t in other.dict.Keys) {
        if (dict.ContainsKey(t))
          return false;
      }
      return true;
    }

    public bool Contains<G>(G t) {
      return t == null ? occurrencesOfNull > 0 : t is T && dict.ContainsKey((T)(object)t);
    }
    public MultiSet<T> Union(MultiSet<T> other) {
      if (dict.Count + occurrencesOfNull == 0)
        return other;
      else if (other.dict.Count + other.occurrencesOfNull == 0)
        return this;
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var r = ImmutableDictionary<T, int>.Empty.ToBuilder();
#else
      var r = new Dictionary<T, int>();
#endif
      foreach (T t in dict.Keys) {
        var i = 0;
        if (!r.TryGetValue(t, out i)) {
          i = 0;
        }
        r[t] = i + dict[t];
      }
      foreach (T t in other.dict.Keys) {
        var i = 0;
        if (!r.TryGetValue(t, out i)) {
          i = 0;
        }
        r[t] = i + other.dict[t];
      }
      return new MultiSet<T>(r, occurrencesOfNull + other.occurrencesOfNull);
    }
    public MultiSet<T> Intersect(MultiSet<T> other) {
      if (dict.Count == 0 && occurrencesOfNull == 0)
        return this;
      else if (other.dict.Count == 0 && other.occurrencesOfNull == 0)
        return other;
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var r = ImmutableDictionary<T, int>.Empty.ToBuilder();
#else
      var r = new Dictionary<T, int>();
#endif
      foreach (T t in dict.Keys) {
        if (other.dict.ContainsKey(t)) {
          r.Add(t, other.dict[t] < dict[t] ? other.dict[t] : dict[t]);
        }
      }
      return new MultiSet<T>(r, other.occurrencesOfNull < occurrencesOfNull ? other.occurrencesOfNull : occurrencesOfNull);
    }
    public MultiSet<T> Difference(MultiSet<T> other) { // \result == this - other
      if (dict.Count == 0 && occurrencesOfNull == 0)
        return this;
      else if (other.dict.Count == 0 && other.occurrencesOfNull == 0)
        return this;
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var r = ImmutableDictionary<T, int>.Empty.ToBuilder();
#else
      var r = new Dictionary<T, int>();
#endif
      foreach (T t in dict.Keys) {
        if (!other.dict.ContainsKey(t)) {
          r.Add(t, dict[t]);
        } else if (other.dict[t] < dict[t]) {
          r.Add(t, dict[t] - other.dict[t]);
        }
      }
      return new MultiSet<T>(r, other.occurrencesOfNull < occurrencesOfNull ? occurrencesOfNull - other.occurrencesOfNull : BigInteger.Zero);
    }

    public IEnumerable<T> Elements {
      get {
        for (var i = BigInteger.Zero; i < occurrencesOfNull; i++) {
          yield return default(T);
        }
        foreach (var item in dict) {
          for (int i = 0; i < item.Value; i++) {
            yield return item.Key;
          }
        }
      }
    }
  }

  public class Map<U, V>
  {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
    readonly ImmutableDictionary<U, V> dict;
#else
    readonly Dictionary<U, V> dict;
#endif
    readonly bool hasNullValue;  // true when "null" is a key of the Map
    readonly V nullValue;  // if "hasNullValue", the value that "null" maps to

#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
    Map(ImmutableDictionary<U, V>.Builder d, bool hasNullValue, V nullValue) {
      dict = d.ToImmutable();
      this.hasNullValue = hasNullValue;
      this.nullValue = nullValue;
    }
    public static readonly Map<U, V> Empty = new Map<U, V>(ImmutableDictionary<U, V>.Empty.ToBuilder(), false, default(V));
#else
    Map(Dictionary<U, V> d, bool hasNullValue, V nullValue) {
      this.dict = d;
      this.hasNullValue = hasNullValue;
      this.nullValue = nullValue;
    }
    public static readonly Map<U, V> Empty = new Map<U, V>(new Dictionary<U, V>(), false, default(V));
#endif

    public static Map<U, V> FromElements(params Pair<U, V>[] values) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var d = ImmutableDictionary<U, V>.Empty.ToBuilder();
#else
      var d = new Dictionary<U, V>(values.Length);
#endif
      var hasNullValue = false;
      var nullValue = default(V);
      foreach (Pair<U, V> p in values) {
        if (p.Car == null) {
          hasNullValue = true;
          nullValue = p.Cdr;
        } else {
          d[p.Car] = p.Cdr;
        }
      }
      return new Map<U, V>(d, hasNullValue, nullValue);
    }
    public static Map<U, V> FromCollection(List<Pair<U, V>> values) {
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
      var d = ImmutableDictionary<U, V>.Empty.ToBuilder();
#else
      var d = new Dictionary<U, V>(values.Count);
#endif
      var hasNullValue = false;
      var nullValue = default(V);
      foreach (Pair<U, V> p in values) {
        if (p.Car == null) {
          hasNullValue = true;
          nullValue = p.Cdr;
        } else {
          d[p.Car] = p.Cdr;
        }
      }
      return new Map<U, V>(d, hasNullValue, nullValue);
    }
    public int Length {
      get { return dict.Count + (hasNullValue ? 1 : 0); }
    }
    public long LongLength {
      get { return dict.Count + (hasNullValue ? 1 : 0); }
    }
    public bool Equals(Map<U, V> other) {
      if (hasNullValue != other.hasNullValue || dict.Count != other.dict.Count) {
        return false;
      } else if (hasNullValue && !Dafny.Helpers.AreEqual(nullValue, other.nullValue)) {
        return false;
      }
      foreach (U u in dict.Keys) {
        V v1 = dict[u];
        V v2;
        if (!other.dict.TryGetValue(u, out v2)) {
          return false; // other dictionary does not contain this element
        }
        if (!Dafny.Helpers.AreEqual(v1, v2)) {
          return false;
        }
      }
      return true;
    }
    public override bool Equals(object other) {
      return other is Map<U, V> && Equals((Map<U, V>)other);
    }
    public override int GetHashCode() {
      var hashCode = 1;
      if (hasNullValue) {
        var key = Dafny.Helpers.GetHashCode(default(U));
        key = (key << 3) | (key >> 29) ^ Dafny.Helpers.GetHashCode(nullValue);
        hashCode = hashCode * (key + 3);
      }
      foreach (var kv in dict) {
        var key = Dafny.Helpers.GetHashCode(kv.Key);
        key = (key << 3) | (key >> 29) ^ Dafny.Helpers.GetHashCode(kv.Value);
        hashCode = hashCode * (key + 3);
      }
      return hashCode;
    }
    public override string ToString() {
      var s = "map[";
      var sep = "";
      if (hasNullValue) {
        s += sep + Dafny.Helpers.ToString(default(U)) + " := " + Dafny.Helpers.ToString(nullValue);
        sep = ", ";
      }
      foreach (var kv in dict) {
        s += sep + Dafny.Helpers.ToString(kv.Key) + " := " + Dafny.Helpers.ToString(kv.Value);
        sep = ", ";
      }
      return s + "]";
    }
    public bool IsDisjointFrom(Map<U, V> other) {
      if (hasNullValue && other.hasNullValue) {
        return false;
      }
      foreach (U u in dict.Keys) {
        if (other.dict.ContainsKey(u))
          return false;
      }
      foreach (U u in other.dict.Keys) {
        if (dict.ContainsKey(u))
          return false;
      }
      return true;
    }
    public bool Contains<G>(G u) {
      return u == null ? hasNullValue : u is U && dict.ContainsKey((U)(object)u);
    }
    public V Select(U index) {
      // evidently, the following will throw some exception if "index" in not a key of the map
      return index == null && hasNullValue ? nullValue : dict[index];
    }
#if DAFNY_USE_SYSTEM_COLLECTIONS_IMMUTABLE
    public Map<U, V> Update(U index, V val) {
      var d = dict.ToBuilder();
      if (index == null) {
        return new Map<U, V>(d, true, val);
      } else {
        d[index] = val;
        return new Map<U, V>(d, hasNullValue, nullValue);
      }
    }
#else
    public Map<U, V> Update(U index, V val) {
      if (index == null) {
        return new Map<U, V>(dict, true, val);
      } else {
        var d = new Dictionary<U, V>(dict);
        d[index] = val;
        return new Map<U, V>(d, hasNullValue, nullValue);
      }
    }
#endif
    public IEnumerable<U> Domain {
      get {
        return dict.Keys;
      }
    }
    public Set<U> Keys
    {
      get
      {
        if (hasNullValue) {
          return Dafny.Set<U>.FromCollectionPlusOne(dict.Keys, default(U));
        } else {
          return Dafny.Set<U>.FromCollection(dict.Keys);
        }
      }
    }
    public Set<V> Values
    {
      get
      {
        if (hasNullValue) {
          return Dafny.Set<V>.FromCollectionPlusOne(dict.Values, nullValue);
        } else {
          return Dafny.Set<V>.FromCollection(dict.Values);
        }
      }
    }
    public Set<@_System.@__tuple_h2<U, V>> Items
    {
      get
      {
        HashSet<@_System.@__tuple_h2<U, V>> result = new HashSet<@_System.@__tuple_h2<U, V>>();
        if (hasNullValue) {
          result.Add(new @_System.@__tuple_h2<U, V>(new @_System.@__tuple_h2____hMake2<U, V>(default(U), nullValue)));
        }
        foreach (KeyValuePair<U, V> kvp in dict) {
          result.Add(new @_System.@__tuple_h2<U,V>(new @_System.@__tuple_h2____hMake2<U, V>(kvp.Key, kvp.Value)));
        }
        return Dafny.Set<@_System.@__tuple_h2<U, V>>.FromCollection(result);
      }
    }
  }

    public class Sequence<T>
  {
    readonly T[] elmts;
    public Sequence(T[] ee) {
      elmts = ee;
    }
    public static Sequence<T> Empty {
      get {
        return new Sequence<T>(new T[0]);
      }
    }
    public static Sequence<T> FromElements(params T[] values) {
      return new Sequence<T>(values);
    }
    public static Sequence<char> FromString(string s) {
      return new Sequence<char>(s.ToCharArray());
    }
    public int Length {
      get { return elmts.Length; }
    }
    public long LongLength {
      get { return elmts.LongLength; }
    }
    public T[] Elements {
      get {
        return elmts;
      }
    }
    public IEnumerable<T> UniqueElements {
      get {
        var st = Set<T>.FromElements(elmts);
        return st.Elements;
      }
    }
    public T Select(ulong index) {
      return elmts[index];
    }
    public T Select(long index) {
      return elmts[index];
    }
    public T Select(uint index) {
      return elmts[index];
    }
    public T Select(int index) {
      return elmts[index];
    }
    public T Select(BigInteger index) {
      return elmts[(int)index];
    }
    public Sequence<T> Update(long index, T t) {
      T[] a = (T[])elmts.Clone();
      a[index] = t;
      return new Sequence<T>(a);
    }
    public Sequence<T> Update(ulong index, T t) {
      return Update((long)index, t);
    }
    public Sequence<T> Update(BigInteger index, T t) {
      return Update((long)index, t);
    }
    public bool Equals(Sequence<T> other) {
      int n = elmts.Length;
      return n == other.elmts.Length && EqualUntil(other, n);
    }
    public override bool Equals(object other) {
      return other is Sequence<T> && Equals((Sequence<T>)other);
    }
    public override int GetHashCode() {
      if (elmts == null || elmts.Length == 0)
        return 0;
      var hashCode = 0;
      for (var i = 0; i < elmts.Length; i++) {
        hashCode = (hashCode << 3) | (hashCode >> 29) ^ Dafny.Helpers.GetHashCode(elmts[i]);
      }
      return hashCode;
    }
    public override string ToString() {
      if (elmts is char[]) {
        var s = "";
        foreach (var t in elmts) {
          s += t.ToString();
        }
        return s;
      } else {
        var s = "[";
        var sep = "";
        foreach (var t in elmts) {
          s += sep + Dafny.Helpers.ToString(t);
          sep = ", ";
        }
        return s + "]";
      }
    }
    bool EqualUntil(Sequence<T> other, int n) {
      for (int i = 0; i < n; i++) {
        if (!Dafny.Helpers.AreEqual(elmts[i], other.elmts[i]))
          return false;
      }
      return true;
    }
    public bool IsProperPrefixOf(Sequence<T> other) {
      int n = elmts.Length;
      return n < other.elmts.Length && EqualUntil(other, n);
    }
    public bool IsPrefixOf(Sequence<T> other) {
      int n = elmts.Length;
      return n <= other.elmts.Length && EqualUntil(other, n);
    }
    public Sequence<T> Concat(Sequence<T> other) {
      if (elmts.Length == 0)
        return other;
      else if (other.elmts.Length == 0)
        return this;
      T[] a = new T[elmts.Length + other.elmts.Length];
      System.Array.Copy(elmts, 0, a, 0, elmts.Length);
      System.Array.Copy(other.elmts, 0, a, elmts.Length, other.elmts.Length);
      return new Sequence<T>(a);
    }
    public bool Contains<G>(G g) {
      if (g == null || g is T) {
        var t = (T)(object)g;
        int n = elmts.Length;
        for (int i = 0; i < n; i++) {
          if (Dafny.Helpers.AreEqual(t, elmts[i]))
            return true;
        }
      }
      return false;
    }
    public Sequence<T> Take(long m) {
      if (elmts.LongLength == m)
        return this;
      T[] a = new T[m];
      System.Array.Copy(elmts, a, m);
      return new Sequence<T>(a);
    }
    public Sequence<T> Take(ulong n) {
      return Take((long)n);
    }
    public Sequence<T> Take(BigInteger n) {
      return Take((long)n);
    }
    public Sequence<T> Drop(long m) {
      if (m == 0)
        return this;
      T[] a = new T[elmts.Length - m];
      System.Array.Copy(elmts, m, a, 0, elmts.Length - m);
      return new Sequence<T>(a);
    }
    public Sequence<T> Drop(ulong n) {
      return Drop((long)n);
    }
    public Sequence<T> Drop(BigInteger n) {
      if (n.IsZero)
        return this;
      return Drop((long)n);
    }
  }
  public struct Pair<A, B>
  {
    public readonly A Car;
    public readonly B Cdr;
    public Pair(A a, B b) {
      this.Car = a;
      this.Cdr = b;
    }
  }
  public partial class Helpers {
    public static bool AreEqual<G>(G a, G b) {
      return a == null ? b == null : a.Equals(b);
    }
    public static int GetHashCode<G>(G g) {
      return g == null ? 1001 : g.GetHashCode();
    }
    public static string ToString<G>(G g) {
      return g == null ? "null" : g.ToString();
    }
    public static System.Predicate<BigInteger> PredicateConverter_byte(System.Predicate<byte> pred) {
      return x => pred((byte)x);
    }
    public static System.Predicate<BigInteger> PredicateConverter_sbyte(System.Predicate<sbyte> pred) {
      return x => pred((sbyte)x);
    }
    public static System.Predicate<BigInteger> PredicateConverter_ushort(System.Predicate<ushort> pred) {
      return x => pred((ushort)x);
    }
    public static System.Predicate<BigInteger> PredicateConverter_short(System.Predicate<short> pred) {
      return x => pred((short)x);
    }
    public static System.Predicate<BigInteger> PredicateConverter_uint(System.Predicate<uint> pred) {
      return x => pred((uint)x);
    }
    public static System.Predicate<BigInteger> PredicateConverter_int(System.Predicate<int> pred) {
      return x => pred((int)x);
    }
    public static System.Predicate<BigInteger> PredicateConverter_ulong(System.Predicate<ulong> pred) {
      return x => pred((ulong)x);
    }
    public static System.Predicate<BigInteger> PredicateConverter_long(System.Predicate<long> pred) {
      return x => pred((long)x);
    }
    // Computing forall/exists quantifiers
    public static bool QuantBool(bool frall, System.Predicate<bool> pred) {
      if (frall) {
        return pred(false) && pred(true);
      } else {
        return pred(false) || pred(true);
      }
    }
    public static bool QuantChar(bool frall, System.Predicate<char> pred) {
      for (int i = 0; i < 0x10000; i++) {
        if (pred((char)i) != frall) { return !frall; }
      }
      return frall;
    }
    public static bool QuantInt(BigInteger lo, BigInteger hi, bool frall, System.Predicate<BigInteger> pred) {
      for (BigInteger i = lo; i < hi; i++) {
        if (pred(i) != frall) { return !frall; }
      }
      return frall;
    }
    public static bool QuantSingle<U>(U u, bool frall, System.Predicate<U> pred) {
      return pred(u);
    }
    public static bool QuantSet<U>(Dafny.Set<U> set, bool frall, System.Predicate<U> pred) {
      foreach (var u in set.Elements) {
        if (pred(u) != frall) { return !frall; }
      }
      return frall;
    }
    public static bool QuantSubSets<U>(Dafny.Set<U> set, bool frall, System.Predicate<Dafny.Set<U>> pred) {
      foreach (var u in set.AllSubsets) {
        if (pred(u) != frall) { return !frall; }
      }
      return frall;
    }
    public static bool QuantMap<U,V>(Dafny.Map<U,V> map, bool frall, System.Predicate<U> pred) {
      foreach (var u in map.Domain) {
        if (pred(u) != frall) { return !frall; }
      }
      return frall;
    }
    public static bool QuantSeq<U>(Dafny.Sequence<U> seq, bool frall, System.Predicate<U> pred) {
      foreach (var u in seq.Elements) {
        if (pred(u) != frall) { return !frall; }
      }
      return frall;
    }
    public static bool QuantDatatype<U>(IEnumerable<U> set, bool frall, System.Predicate<U> pred) {
      foreach (var u in set) {
        if (pred(u) != frall) { return !frall; }
      }
      return frall;
    }
    // Enumerating other collections
    public delegate Dafny.Set<T> ComprehensionDelegate<T>();
    public delegate Dafny.Map<U, V> MapComprehensionDelegate<U, V>();
    public static IEnumerable<bool> AllBooleans {
      get {
        yield return false;
        yield return true;
      }
    }
    public static IEnumerable<char> AllChars {
      get {
        for (int i = 0; i < 0x10000; i++) {
          yield return (char)i;
        }
      }
    }
    public static IEnumerable<BigInteger> AllIntegers {
      get {
        yield return new BigInteger(0);
        for (var j = new BigInteger(1);; j++) {
          yield return j;
          yield return -j;
        }
      }
    }
    public static IEnumerable<BigInteger> IntegerRange(Nullable<BigInteger> lo, Nullable<BigInteger> hi) {
      if (lo == null) {
        for (var j = (BigInteger)hi; true; ) {
          j--;
          yield return j;
        }
      } else if (hi == null) {
        for (var j = (BigInteger)lo; true; j++) {
          yield return j;
        }
      } else {
        for (var j = (BigInteger)lo; j < hi; j++) {
          yield return j;
        }
      }
    }
    public static IEnumerable<T> SingleValue<T>(T e) {
      yield return e;
    }
    // pre: b != 0
    // post: result == a/b, as defined by Euclidean Division (http://en.wikipedia.org/wiki/Modulo_operation)
    public static sbyte EuclideanDivision_sbyte(sbyte a, sbyte b) {
      return (sbyte)EuclideanDivision_int(a, b);
    }
    public static short EuclideanDivision_short(short a, short b) {
      return (short)EuclideanDivision_int(a, b);
    }
    public static int EuclideanDivision_int(int a, int b) {
      if (0 <= a) {
        if (0 <= b) {
          // +a +b: a/b
          return (int)(((uint)(a)) / ((uint)(b)));
        } else {
          // +a -b: -(a/(-b))
          return -((int)(((uint)(a)) / ((uint)(unchecked(-b)))));
        }
      } else {
        if (0 <= b) {
          // -a +b: -((-a-1)/b) - 1
          return -((int)(((uint)(-(a + 1))) / ((uint)(b)))) - 1;
        } else {
          // -a -b: ((-a-1)/(-b)) + 1
          return ((int)(((uint)(-(a + 1))) / ((uint)(unchecked(-b))))) + 1;
        }
      }
    }
    public static long EuclideanDivision_long(long a, long b) {
      if (0 <= a) {
        if (0 <= b) {
          // +a +b: a/b
          return (long)(((ulong)(a)) / ((ulong)(b)));
        } else {
          // +a -b: -(a/(-b))
          return -((long)(((ulong)(a)) / ((ulong)(unchecked(-b)))));
        }
      } else {
        if (0 <= b) {
          // -a +b: -((-a-1)/b) - 1
          return -((long)(((ulong)(-(a + 1))) / ((ulong)(b)))) - 1;
        } else {
          // -a -b: ((-a-1)/(-b)) + 1
          return ((long)(((ulong)(-(a + 1))) / ((ulong)(unchecked(-b))))) + 1;
        }
      }
    }
    public static BigInteger EuclideanDivision(BigInteger a, BigInteger b) {
      if (0 <= a.Sign) {
        if (0 <= b.Sign) {
          // +a +b: a/b
          return BigInteger.Divide(a, b);
        } else {
          // +a -b: -(a/(-b))
          return BigInteger.Negate(BigInteger.Divide(a, BigInteger.Negate(b)));
        }
      } else {
        if (0 <= b.Sign) {
          // -a +b: -((-a-1)/b) - 1
          return BigInteger.Negate(BigInteger.Divide(BigInteger.Negate(a) - 1, b)) - 1;
        } else {
          // -a -b: ((-a-1)/(-b)) + 1
          return BigInteger.Divide(BigInteger.Negate(a) - 1, BigInteger.Negate(b)) + 1;
        }
      }
    }
    // pre: b != 0
    // post: result == a%b, as defined by Euclidean Division (http://en.wikipedia.org/wiki/Modulo_operation)
    public static sbyte EuclideanModulus_sbyte(sbyte a, sbyte b) {
      return (sbyte)EuclideanModulus_int(a, b);
    }
    public static short EuclideanModulus_short(short a, short b) {
      return (short)EuclideanModulus_int(a, b);
    }
    public static int EuclideanModulus_int(int a, int b) {
      uint bp = (0 <= b) ? (uint)b : (uint)(unchecked(-b));
      if (0 <= a) {
        // +a: a % b'
        return (int)(((uint)a) % bp);
      } else {
        // c = ((-a) % b')
        // -a: b' - c if c > 0
        // -a: 0 if c == 0
        uint c = ((uint)(unchecked(-a))) % bp;
        return (int)(c == 0 ? c : bp - c);
      }
    }
    public static long EuclideanModulus_long(long a, long b) {
      ulong bp = (0 <= b) ? (ulong)b : (ulong)(unchecked(-b));
      if (0 <= a) {
        // +a: a % b'
        return (long)(((ulong)a) % bp);
      } else {
        // c = ((-a) % b')
        // -a: b' - c if c > 0
        // -a: 0 if c == 0
        ulong c = ((ulong)(unchecked(-a))) % bp;
        return (long)(c == 0 ? c : bp - c);
      }
    }
    public static BigInteger EuclideanModulus(BigInteger a, BigInteger b) {
      var bp = BigInteger.Abs(b);
      if (0 <= a.Sign) {
        // +a: a % b'
        return BigInteger.Remainder(a, bp);
      } else {
        // c = ((-a) % b')
        // -a: b' - c if c > 0
        // -a: 0 if c == 0
        var c = BigInteger.Remainder(BigInteger.Negate(a), bp);
        return c.IsZero ? c : BigInteger.Subtract(bp, c);
      }
    }
    public static Sequence<T> SeqFromArray<T>(T[] array) {
      return new Sequence<T>((T[])array.Clone());
    }
    // In .NET version 4.5, it it possible to mark a method with "AggressiveInlining", which says to inline the
    // method if possible.  Method "ExpressionSequence" would be a good candidate for it:
    // [System.Runtime.CompilerServices.MethodImpl(System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
    public static U ExpressionSequence<T, U>(T t, U u)
    {
      return u;
    }

    public static U Let<T, U>(T t, Func<T,U> f) {
      return f(t);
    }

    public delegate Result Function<Input,Result>(Input input);

    public static A Id<A>(A a) {
      return a;
    }

    public static bool BigOrdinal_IsLimit(BigInteger ord) {
      return ord == 0;
    }
    public static bool BigOrdinal_IsSucc(BigInteger ord) {
      return 0 < ord;
    }
    public static BigInteger BigOrdinal_Offset(BigInteger ord) {
      return ord;
    }
    public static bool BigOrdinal_IsNat(BigInteger ord) {
      return true;  // at run time, every ORDINAL is a natural number
    }
  }

  public struct BigRational
  {
    public static readonly BigRational ZERO = new BigRational(0);

    // We need to deal with the special case "num == 0 && den == 0", because
    // that's what C#'s default struct constructor will produce for BigRational. :(
    // To deal with it, we ignore "den" when "num" is 0.
    BigInteger num, den;  // invariant 1 <= den || (num == 0 && den == 0)
    public override string ToString() {
      if (num.IsZero || den.IsOne) {
        return string.Format("{0}.0", num);
      } else {
        return string.Format("({0}.0 / {1}.0)", num, den);
      }
    }
    public BigRational(int n) {
      num = new BigInteger(n);
      den = BigInteger.One;
    }
    public BigRational(BigInteger n, BigInteger d) {
      // requires 1 <= d
      num = n;
      den = d;
    }
    public BigInteger ToBigInteger() {
      if (num.IsZero || den.IsOne) {
        return num;
      } else if (0 < num.Sign) {
        return num / den;
      } else {
        return (num - den + 1) / den;
      }
    }
    /// <summary>
    /// Returns values such that aa/dd == a and bb/dd == b.
    /// </summary>
    private static void Normalize(BigRational a, BigRational b, out BigInteger aa, out BigInteger bb, out BigInteger dd) {
      if (a.num.IsZero) {
        aa = a.num;
        bb = b.num;
        dd = b.den;
      } else if (b.num.IsZero) {
        aa = a.num;
        dd = a.den;
        bb = b.num;
      } else {
        var gcd = BigInteger.GreatestCommonDivisor(a.den, b.den);
        var xx = a.den / gcd;
        var yy = b.den / gcd;
        // We now have a == a.num / (xx * gcd) and b == b.num / (yy * gcd).
        aa = a.num * yy;
        bb = b.num * xx;
        dd = a.den * yy;
      }
    }
    public int CompareTo(BigRational that) {
      // simple things first
      int asign = this.num.Sign;
      int bsign = that.num.Sign;
      if (asign < 0 && 0 <= bsign) {
        return -1;
      } else if (asign <= 0 && 0 < bsign) {
        return -1;
      } else if (bsign < 0 && 0 <= asign) {
        return 1;
      } else if (bsign <= 0 && 0 < asign) {
        return 1;
      }
      BigInteger aa, bb, dd;
      Normalize(this, that, out aa, out bb, out dd);
      return aa.CompareTo(bb);
    }
    public override int GetHashCode() {
      return num.GetHashCode() + 29 * den.GetHashCode();
    }
    public override bool Equals(object obj) {
      if (obj is BigRational) {
        return this == (BigRational)obj;
      } else {
        return false;
      }
    }
    public static bool operator ==(BigRational a, BigRational b) {
      return a.CompareTo(b) == 0;
    }
    public static bool operator !=(BigRational a, BigRational b) {
      return a.CompareTo(b) != 0;
    }
    public static bool operator >(BigRational a, BigRational b) {
      return a.CompareTo(b) > 0;
    }
    public static bool operator >=(BigRational a, BigRational b) {
      return a.CompareTo(b) >= 0;
    }
    public static bool operator <(BigRational a, BigRational b) {
      return a.CompareTo(b) < 0;
    }
    public static bool operator <=(BigRational a, BigRational b) {
      return a.CompareTo(b) <= 0;
    }
    public static BigRational operator +(BigRational a, BigRational b) {
      BigInteger aa, bb, dd;
      Normalize(a, b, out aa, out bb, out dd);
      return new BigRational(aa + bb, dd);
    }
    public static BigRational operator -(BigRational a, BigRational b) {
      BigInteger aa, bb, dd;
      Normalize(a, b, out aa, out bb, out dd);
      return new BigRational(aa - bb, dd);
    }
    public static BigRational operator -(BigRational a) {
      return new BigRational(-a.num, a.den);
    }
    public static BigRational operator *(BigRational a, BigRational b) {
      return new BigRational(a.num * b.num, a.den * b.den);
    }
    public static BigRational operator /(BigRational a, BigRational b) {
      // Compute the reciprocal of b
      BigRational bReciprocal;
      if (0 < b.num.Sign) {
        bReciprocal = new BigRational(b.den, b.num);
      } else {
        // this is the case b.num < 0
        bReciprocal = new BigRational(-b.den, -b.num);
      }
      return a * bReciprocal;
    }
  }
}

namespace @_System
{

  public abstract class Base___tuple_h2<@T0, @T1> { }
  public class __tuple_h2____hMake2<@T0, @T1> : Base___tuple_h2<@T0, @T1>
  {
    public readonly @T0 @_0;
    public readonly @T1 @_1;
    public __tuple_h2____hMake2(@T0 @_0, @T1 @_1)
    {
      this.@_0 = @_0;
      this.@_1 = @_1;
    }
    public override bool Equals(object other)
    {
      var oth = other as _System.@__tuple_h2____hMake2<@T0, @T1>;
      return oth != null && this.@_0.Equals(oth.@_0) && Dafny.Helpers.AreEqual(this.@_1, oth.@_1);
    }
    public override int GetHashCode()
    {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@_0));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@_1));
      return (int)hash;
    }
    public override string ToString()
    {
      string s = "";
      s += "(";
      s += Dafny.Helpers.ToString(this.@_0);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@_1);
      s += ")";
      return s;
    }
  }
  public struct @__tuple_h2<@T0, @T1>
  {
    Base___tuple_h2<@T0, @T1> _d;
    public Base___tuple_h2<@T0, @T1> _D
    {
      get
      {
        if (_d == null) {
          _d = Default;
        }
        return _d;
      }
    }
    public @__tuple_h2(Base___tuple_h2<@T0, @T1> d) { this._d = d; }
    static Base___tuple_h2<@T0, @T1> theDefault;
    public static Base___tuple_h2<@T0, @T1> Default
    {
      get
      {
        if (theDefault == null) {
          theDefault = new _System.@__tuple_h2____hMake2<@T0, @T1>(default(@T0), default(@T1));
        }
        return theDefault;
      }
    }
    public override bool Equals(object other)
    {
      return other is @__tuple_h2<@T0, @T1> && _D.Equals(((@__tuple_h2<@T0, @T1>)other)._D);
    }
    public override int GetHashCode() { return _D.GetHashCode(); }
    public override string ToString() { return _D.ToString(); }
    public bool is____hMake2 { get { return _D is __tuple_h2____hMake2<@T0, @T1>; } }
    public @T0 dtor__0 { get { return ((__tuple_h2____hMake2<@T0, @T1>)_D).@_0; } }
    public @T1 dtor__1 { get { return ((__tuple_h2____hMake2<@T0, @T1>)_D).@_1; } }
  }
} // end of namespace _System
namespace Dafny {
  internal class ArrayHelpers {
      public static T[] InitNewArray1<T>(T z, BigInteger size0) {
        int s0 = (int)size0;
        T[] a = new T[s0];
        for (int i0 = 0; i0 < s0; i0++)
          a[i0] = z;
        return a;
      }
  }
}
namespace @_System {



  public abstract class Base___tuple_h0 { }
  public class __tuple_h0____hMake0 : Base___tuple_h0 {
    public __tuple_h0____hMake0() {
    }
    public override bool Equals(object other) {
      var oth = other as _System.@__tuple_h0____hMake0;
      return oth != null;
    }
    public override int GetHashCode() {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      return (int) hash;
    }
    public override string ToString() {
      string s = "";
      return s;
    }
  }
  public struct @__tuple_h0 {
    Base___tuple_h0 _d;
    public Base___tuple_h0 _D {
      get {
        if (_d == null) {
          _d = Default;
        }
        return _d;
      }
    }
    public @__tuple_h0(Base___tuple_h0 d) { this._d = d; }
    static Base___tuple_h0 theDefault;
    public static Base___tuple_h0 Default {
      get {
        if (theDefault == null) {
          theDefault = new _System.@__tuple_h0____hMake0();
        }
        return theDefault;
      }
    }
    public override bool Equals(object other) {
      return other is @__tuple_h0 && _D.Equals(((@__tuple_h0)other)._D);
    }
    public override int GetHashCode() { return _D.GetHashCode(); }
    public override string ToString() { return _D.ToString(); }
    public bool is____hMake0 { get { return _D is __tuple_h0____hMake0; } }
    public static System.Collections.Generic.IEnumerable<@__tuple_h0> AllSingletonConstructors {
      get {
        yield return new @__tuple_h0(new __tuple_h0____hMake0());
        yield break;
      }
    }
  }
} // end of namespace _System
namespace @_0_Utils_Compile {


  public partial class @__default {
  }
} // end of namespace _0_Utils_Compile
namespace @_0_QCSP__Instance_Compile {








  public abstract class Base_Structure<@T> { }
  public class Structure_Structure<@T> : Base_Structure<@T> {
    public readonly Dafny.Map<Dafny.Sequence<char>,BigInteger> @Sig;
    public readonly Dafny.Set<@T> @Dom;
    public readonly Dafny.Map<Dafny.Sequence<char>,Dafny.Set<Dafny.Sequence<@T>>> @I;
    public Structure_Structure(Dafny.Map<Dafny.Sequence<char>,BigInteger> @Sig, Dafny.Set<@T> @Dom, Dafny.Map<Dafny.Sequence<char>,Dafny.Set<Dafny.Sequence<@T>>> @I) {
      this.@Sig = @Sig;
      this.@Dom = @Dom;
      this.@I = @I;
    }
    public override bool Equals(object other) {
      var oth = other as _0_QCSP__Instance_Compile.@Structure_Structure<@T>;
      return oth != null && Dafny.Helpers.AreEqual(this.@Sig, oth.@Sig) && Dafny.Helpers.AreEqual(this.@Dom, oth.@Dom) && Dafny.Helpers.AreEqual(this.@I, oth.@I);
    }
    public override int GetHashCode() {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@Sig));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@Dom));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@I));
      return (int) hash;
    }
    public override string ToString() {
      string s = "_0_QCSP__Instance_Compile.Structure.Structure";
      s += "(";
      s += Dafny.Helpers.ToString(this.@Sig);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@Dom);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@I);
      s += ")";
      return s;
    }
  }
  public struct @Structure<@T> {
    Base_Structure<@T> _d;
    public Base_Structure<@T> _D {
      get {
        if (_d == null) {
          _d = Default;
        }
        return _d;
      }
    }
    public @Structure(Base_Structure<@T> d) { this._d = d; }
    static Base_Structure<@T> theDefault;
    public static Base_Structure<@T> Default {
      get {
        if (theDefault == null) {
          theDefault = new _0_QCSP__Instance_Compile.@Structure_Structure<@T>(Dafny.Map<Dafny.Sequence<char>,BigInteger>.Empty, Dafny.Set<@T>.Empty, Dafny.Map<Dafny.Sequence<char>,Dafny.Set<Dafny.Sequence<@T>>>.Empty);
        }
        return theDefault;
      }
    }
    public override bool Equals(object other) {
      return other is @Structure<@T> && _D.Equals(((@Structure<@T>)other)._D);
    }
    public override int GetHashCode() { return _D.GetHashCode(); }
    public override string ToString() { return _D.ToString(); }
    public bool is_Structure { get { return _D is Structure_Structure<@T>; } }
    public Dafny.Map<Dafny.Sequence<char>,BigInteger> dtor_Sig { get { var d = _D; return ((Structure_Structure<@T>)d).@Sig; } }
    public Dafny.Set<@T> dtor_Dom { get { var d = _D; return ((Structure_Structure<@T>)d).@Dom; } }
    public Dafny.Map<Dafny.Sequence<char>,Dafny.Set<Dafny.Sequence<@T>>> dtor_I { get { var d = _D; return ((Structure_Structure<@T>)d).@I; } }
  }

  public abstract class Base_Formula { }
  public class Formula_Atom : Base_Formula {
    public readonly Dafny.Sequence<char> @rel;
    public readonly Dafny.Sequence<Dafny.Sequence<char>> @par;
    public Formula_Atom(Dafny.Sequence<char> @rel, Dafny.Sequence<Dafny.Sequence<char>> @par) {
      this.@rel = @rel;
      this.@par = @par;
    }
    public override bool Equals(object other) {
      var oth = other as _0_QCSP__Instance_Compile.@Formula_Atom;
      return oth != null && Dafny.Helpers.AreEqual(this.@rel, oth.@rel) && Dafny.Helpers.AreEqual(this.@par, oth.@par);
    }
    public override int GetHashCode() {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@rel));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@par));
      return (int) hash;
    }
    public override string ToString() {
      string s = "_0_QCSP__Instance_Compile.Formula.Atom";
      s += "(";
      s += Dafny.Helpers.ToString(this.@rel);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@par);
      s += ")";
      return s;
    }
  }
  public class Formula_And : Base_Formula {
    public readonly @_0_QCSP__Instance_Compile.@Formula @_0;
    public readonly @_0_QCSP__Instance_Compile.@Formula @_1;
    public Formula_And(@_0_QCSP__Instance_Compile.@Formula @_0, @_0_QCSP__Instance_Compile.@Formula @_1) {
      this.@_0 = @_0;
      this.@_1 = @_1;
    }
    public override bool Equals(object other) {
      var oth = other as _0_QCSP__Instance_Compile.@Formula_And;
      return oth != null && Dafny.Helpers.AreEqual(this.@_0, oth.@_0) && Dafny.Helpers.AreEqual(this.@_1, oth.@_1);
    }
    public override int GetHashCode() {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@_0));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@_1));
      return (int) hash;
    }
    public override string ToString() {
      string s = "_0_QCSP__Instance_Compile.Formula.And";
      s += "(";
      s += Dafny.Helpers.ToString(this.@_0);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@_1);
      s += ")";
      return s;
    }
  }
  public class Formula_Forall : Base_Formula {
    public readonly Dafny.Sequence<char> @x;
    public readonly @_0_QCSP__Instance_Compile.@Formula @Body;
    public Formula_Forall(Dafny.Sequence<char> @x, @_0_QCSP__Instance_Compile.@Formula @Body) {
      this.@x = @x;
      this.@Body = @Body;
    }
    public override bool Equals(object other) {
      var oth = other as _0_QCSP__Instance_Compile.@Formula_Forall;
      return oth != null && Dafny.Helpers.AreEqual(this.@x, oth.@x) && Dafny.Helpers.AreEqual(this.@Body, oth.@Body);
    }
    public override int GetHashCode() {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@x));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@Body));
      return (int) hash;
    }
    public override string ToString() {
      string s = "_0_QCSP__Instance_Compile.Formula.Forall";
      s += "(";
      s += Dafny.Helpers.ToString(this.@x);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@Body);
      s += ")";
      return s;
    }
  }
  public class Formula_Exists : Base_Formula {
    public readonly Dafny.Sequence<char> @x;
    public readonly @_0_QCSP__Instance_Compile.@Formula @Body;
    public Formula_Exists(Dafny.Sequence<char> @x, @_0_QCSP__Instance_Compile.@Formula @Body) {
      this.@x = @x;
      this.@Body = @Body;
    }
    public override bool Equals(object other) {
      var oth = other as _0_QCSP__Instance_Compile.@Formula_Exists;
      return oth != null && Dafny.Helpers.AreEqual(this.@x, oth.@x) && Dafny.Helpers.AreEqual(this.@Body, oth.@Body);
    }
    public override int GetHashCode() {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@x));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@Body));
      return (int) hash;
    }
    public override string ToString() {
      string s = "_0_QCSP__Instance_Compile.Formula.Exists";
      s += "(";
      s += Dafny.Helpers.ToString(this.@x);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@Body);
      s += ")";
      return s;
    }
  }
  public struct @Formula {
    Base_Formula _d;
    public Base_Formula _D {
      get {
        if (_d == null) {
          _d = Default;
        }
        return _d;
      }
    }
    public @Formula(Base_Formula d) { this._d = d; }
    static Base_Formula theDefault;
    public static Base_Formula Default {
      get {
        if (theDefault == null) {
          theDefault = new _0_QCSP__Instance_Compile.@Formula_Atom(Dafny.Sequence<char>.Empty, Dafny.Sequence<Dafny.Sequence<char>>.Empty);
        }
        return theDefault;
      }
    }
    public override bool Equals(object other) {
      return other is @Formula && _D.Equals(((@Formula)other)._D);
    }
    public override int GetHashCode() { return _D.GetHashCode(); }
    public override string ToString() { return _D.ToString(); }
    public bool is_Atom { get { return _D is Formula_Atom; } }
    public bool is_And { get { return _D is Formula_And; } }
    public bool is_Forall { get { return _D is Formula_Forall; } }
    public bool is_Exists { get { return _D is Formula_Exists; } }
    public Dafny.Sequence<char> dtor_rel { get { var d = _D; return ((Formula_Atom)d).@rel; } }
    public Dafny.Sequence<Dafny.Sequence<char>> dtor_par { get { var d = _D; return ((Formula_Atom)d).@par; } }
    public @_0_QCSP__Instance_Compile.@Formula dtor__0 { get { var d = _D; return ((Formula_And)d).@_0; } }
    public @_0_QCSP__Instance_Compile.@Formula dtor__1 { get { var d = _D; return ((Formula_And)d).@_1; } }
    public Dafny.Sequence<char> dtor_x { get { var d = _D; if (d is Formula_Forall) { return ((Formula_Forall)d).@x; } return ((Formula_Exists)d).@x; } }
    public @_0_QCSP__Instance_Compile.@Formula dtor_Body { get { var d = _D; if (d is Formula_Forall) { return ((Formula_Forall)d).@Body; } return ((Formula_Exists)d).@Body; } }
  }


  public partial class @__default {
    public static Dafny.Sequence<@V> @HOmap<@U,@V>(Dafny.Map<@U,@V> @f, Dafny.Sequence<@U> @xs) {
      if ((@xs).@Equals(Dafny.Sequence<@U>.FromElements())) {
        return Dafny.Sequence<@V>.FromElements();
      } else {
        return (Dafny.Sequence<@V>.FromElements((@f).Select((@xs).Select(new BigInteger(0))))).@Concat(@_0_QCSP__Instance_Compile.@__default.@HOmap<@U,@V>(@f, (@xs).Drop(new BigInteger(1))));
      }
    }
    public static Dafny.Map<Dafny.Sequence<char>,@T> @projectVal<@T>(Dafny.Map<Dafny.Sequence<char>,@T> @f, Dafny.Set<Dafny.Sequence<char>> @U) {
      return ((Dafny.Helpers.MapComprehensionDelegate<Dafny.Sequence<char>,@T>)delegate() { var _coll0 = new System.Collections.Generic.List<Dafny.Pair<Dafny.Sequence<char>,@T>>(); foreach (var @_17165_s in (@U).Elements) { if ((@U).@Contains(@_17165_s)) { _coll0.Add(new Dafny.Pair<Dafny.Sequence<char>,@T>(@_17165_s,(@f).Select(@_17165_s))); }}return Dafny.Map<Dafny.Sequence<char>,@T>.FromCollection(_coll0); })();
    }
  }
} // end of namespace _0_QCSP__Instance_Compile
namespace @_0_Proof__System_Compile {





  public abstract class Base_Judgement<@T> { }
  public class Judgement_J<@T> : Base_Judgement<@T> {
    public readonly Dafny.Sequence<BigInteger> @i;
    public readonly Dafny.Set<Dafny.Sequence<char>> @V;
    public readonly Dafny.Set<Dafny.Map<Dafny.Sequence<char>,@T>> @F;
    public Judgement_J(Dafny.Sequence<BigInteger> @i, Dafny.Set<Dafny.Sequence<char>> @V, Dafny.Set<Dafny.Map<Dafny.Sequence<char>,@T>> @F) {
      this.@i = @i;
      this.@V = @V;
      this.@F = @F;
    }
    public override bool Equals(object other) {
      var oth = other as _0_Proof__System_Compile.@Judgement_J<@T>;
      return oth != null && Dafny.Helpers.AreEqual(this.@i, oth.@i) && Dafny.Helpers.AreEqual(this.@V, oth.@V) && Dafny.Helpers.AreEqual(this.@F, oth.@F);
    }
    public override int GetHashCode() {
      ulong hash = 5381;
      hash = ((hash << 5) + hash) + 0;
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@i));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@V));
      hash = ((hash << 5) + hash) + ((ulong)Dafny.Helpers.GetHashCode(this.@F));
      return (int) hash;
    }
    public override string ToString() {
      string s = "_0_Proof__System_Compile.Judgement.J";
      s += "(";
      s += Dafny.Helpers.ToString(this.@i);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@V);
      s += ", ";
      s += Dafny.Helpers.ToString(this.@F);
      s += ")";
      return s;
    }
  }
  public struct @Judgement<@T> {
    Base_Judgement<@T> _d;
    public Base_Judgement<@T> _D {
      get {
        if (_d == null) {
          _d = Default;
        }
        return _d;
      }
    }
    public @Judgement(Base_Judgement<@T> d) { this._d = d; }
    static Base_Judgement<@T> theDefault;
    public static Base_Judgement<@T> Default {
      get {
        if (theDefault == null) {
          theDefault = new _0_Proof__System_Compile.@Judgement_J<@T>(Dafny.Sequence<BigInteger>.Empty, Dafny.Set<Dafny.Sequence<char>>.Empty, Dafny.Set<Dafny.Map<Dafny.Sequence<char>,@T>>.Empty);
        }
        return theDefault;
      }
    }
    public override bool Equals(object other) {
      return other is @Judgement<@T> && _D.Equals(((@Judgement<@T>)other)._D);
    }
    public override int GetHashCode() { return _D.GetHashCode(); }
    public override string ToString() { return _D.ToString(); }
    public bool is_J { get { return _D is Judgement_J<@T>; } }
    public Dafny.Sequence<BigInteger> dtor_i { get { var d = _D; return ((Judgement_J<@T>)d).@i; } }
    public Dafny.Set<Dafny.Sequence<char>> dtor_V { get { var d = _D; return ((Judgement_J<@T>)d).@V; } }
    public Dafny.Set<Dafny.Map<Dafny.Sequence<char>,@T>> dtor_F { get { var d = _D; return ((Judgement_J<@T>)d).@F; } }
  }

  public partial class @__default {
    public static @_0_QCSP__Instance_Compile.@Formula @FoI(Dafny.Sequence<BigInteger> @s, @_0_QCSP__Instance_Compile.@Formula @phi, Dafny.Map<Dafny.Sequence<char>,BigInteger> @Sig) {
      if ((@s).@Equals(Dafny.Sequence<BigInteger>.FromElements())) {
        return @phi;
      } else {
        @_0_QCSP__Instance_Compile.@Formula _source0 = @phi;
        if (_source0.is_And) {
          @_0_QCSP__Instance_Compile.@Formula @_17166_phi = ((_0_QCSP__Instance_Compile.@Formula_And)_source0._D).@_0;
          @_0_QCSP__Instance_Compile.@Formula @_17167_psi = ((_0_QCSP__Instance_Compile.@Formula_And)_source0._D).@_1;
          if (((@s).Select(new BigInteger(0))).@Equals(new BigInteger(0))) {
            return @_0_Proof__System_Compile.@__default.@FoI((@s).Drop(new BigInteger(1)), @_17166_phi, @Sig);
          } else {
            return @_0_Proof__System_Compile.@__default.@FoI((@s).Drop(new BigInteger(1)), @_17167_psi, @Sig);
          }
        } else if (_source0.is_Forall) {
          Dafny.Sequence<char> @_17168_x = ((_0_QCSP__Instance_Compile.@Formula_Forall)_source0._D).@x;
          @_0_QCSP__Instance_Compile.@Formula @_17169_phi = ((_0_QCSP__Instance_Compile.@Formula_Forall)_source0._D).@Body;
          return @_0_Proof__System_Compile.@__default.@FoI((@s).Drop(new BigInteger(1)), @_17169_phi, @Sig);
        } else if (true) {
          Dafny.Sequence<char> @_17170_x = ((_0_QCSP__Instance_Compile.@Formula_Exists)_source0._D).@x;
          @_0_QCSP__Instance_Compile.@Formula @_17171_phi = ((_0_QCSP__Instance_Compile.@Formula_Exists)_source0._D).@Body;
          return @_0_Proof__System_Compile.@__default.@FoI((@s).Drop(new BigInteger(1)), @_17171_phi, @Sig);
        }
      }
    }
  }
} // end of namespace _0_Proof__System_Compile
namespace @_0_PS__Completeness_Compile {





  public partial class @__default {
    public static @_0_Proof__System_Compile.@Judgement<@T> @projection<@T>(@_0_Proof__System_Compile.@Judgement<@T> @j, Dafny.Set<Dafny.Sequence<char>> @U, @_0_QCSP__Instance_Compile.@Formula @phi, @_0_QCSP__Instance_Compile.@Structure<@T> @B) {
      return new @_0_Proof__System_Compile.@Judgement<@T>(new _0_Proof__System_Compile.@Judgement_J<@T>((@j).@dtor_i, @U, ((Dafny.Helpers.ComprehensionDelegate<Dafny.Map<Dafny.Sequence<char>,@T>>)delegate() { var _coll1 = new System.Collections.Generic.List<Dafny.Map<Dafny.Sequence<char>,@T>>(); foreach (var @_set_compr_0 in ((@j).@dtor_F).Elements) { Dafny.Map<Dafny.Sequence<char>,@T> @_17172_f = (Dafny.Map<Dafny.Sequence<char>,@T>)_set_compr_0; if (((@j).@dtor_F).@Contains(@_17172_f)) {_coll1.Add(@_0_QCSP__Instance_Compile.@__default.@projectVal<@T>(@_17172_f, @U)); }}return Dafny.Set<Dafny.Map<Dafny.Sequence<char>,@T>>.FromCollection(_coll1); })()));
    }
  }
} // end of namespace _0_PS__Completeness_Compile
namespace @__default {


  public partial class @__default {
  }



} // end of namespace @__default
